package com.minecolonies.coremod.entity;

import com.minecolonies.coremod.MineColonies;
import com.minecolonies.coremod.achievements.ModAchievements;
import com.minecolonies.coremod.client.render.RenderBipedCitizen;
import com.minecolonies.coremod.colony.*;
import com.minecolonies.coremod.colony.buildings.AbstractBuildingWorker;
import com.minecolonies.coremod.colony.buildings.BuildingFarmer;
import com.minecolonies.coremod.colony.buildings.BuildingHome;
import com.minecolonies.coremod.colony.jobs.*;
import com.minecolonies.coremod.configuration.Configurations;
import com.minecolonies.coremod.entity.ai.basic.AbstractEntityAIInteract;
import com.minecolonies.coremod.entity.ai.minimal.*;
import com.minecolonies.coremod.entity.pathfinding.PathNavigate;
import com.minecolonies.coremod.entity.pathfinding.WalkToProxy;
import com.minecolonies.coremod.inventory.InventoryCitizen;
import com.minecolonies.coremod.lib.Constants;
import com.minecolonies.coremod.network.messages.BlockParticleEffectMessage;
import com.minecolonies.coremod.util.*;
import net.minecraft.block.Block;
import net.minecraft.block.material.Material;
import net.minecraft.block.state.IBlockState;
import net.minecraft.entity.*;
import net.minecraft.entity.ai.*;
import net.minecraft.entity.item.EntityItem;
import net.minecraft.entity.item.EntityXPOrb;
import net.minecraft.entity.monster.EntityEnderman;
import net.minecraft.entity.monster.EntityGuardian;
import net.minecraft.entity.monster.EntityMob;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.init.Blocks;
import net.minecraft.init.SoundEvents;
import net.minecraft.inventory.EntityEquipmentSlot;
import net.minecraft.item.Item;
import net.minecraft.item.ItemArmor;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.network.datasync.DataParameter;
import net.minecraft.network.datasync.DataSerializers;
import net.minecraft.network.datasync.EntityDataManager;
import net.minecraft.util.*;
import net.minecraft.util.math.AxisAlignedBB;
import net.minecraft.util.math.BlockPos;
import net.minecraft.util.math.MathHelper;
import net.minecraft.util.text.TextComponentString;
import net.minecraft.util.text.TextComponentTranslation;
import net.minecraft.world.World;
import net.minecraftforge.fml.common.network.NetworkRegistry;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.lang.reflect.Field;
import java.util.*;

/**
 * The Class used to represent the citizen entities.
 */
public class EntityCitizen extends EntityAgeable implements INpc
{
    private static final DataParameter<Integer> DATA_TEXTURE         = EntityDataManager.<Integer>func_187226_a(EntityCitizen.class, DataSerializers.field_187192_b);
    private static final DataParameter<Integer> DATA_LEVEL           = EntityDataManager.<Integer>func_187226_a(EntityCitizen.class, DataSerializers.field_187192_b);
    private static final DataParameter<Integer> DATA_IS_FEMALE       = EntityDataManager.<Integer>func_187226_a(EntityCitizen.class, DataSerializers.field_187192_b);
    private static final DataParameter<Integer> DATA_COLONY_ID       = EntityDataManager.<Integer>func_187226_a(EntityCitizen.class, DataSerializers.field_187192_b);
    private static final DataParameter<Integer> DATA_CITIZEN_ID      = EntityDataManager.<Integer>func_187226_a(EntityCitizen.class, DataSerializers.field_187192_b);
    private static final DataParameter<String>  DATA_MODEL           = EntityDataManager.<String>func_187226_a(EntityCitizen.class, DataSerializers.field_187194_d);
    private static final DataParameter<String>  DATA_RENDER_METADATA = EntityDataManager.<String>func_187226_a(EntityCitizen.class, DataSerializers.field_187194_d);

    /**
     * The movement speed for the citizen to run away.
     */
    private static final int MOVE_AWAY_SPEED = 2;

    /**
     * The range for the citizen to move away.
     */
    private static final int MOVE_AWAY_RANGE = 6;

    /**
     * Number of ticks to heal the citizens.
     */
    private static final int HEAL_CITIZENS_AFTER = 100;

    /**
     * Tag's to save data to NBT.
     */
    private static final String TAG_COLONY_ID      = "colony";
    private static final String TAG_CITIZEN        = "citizen";
    private static final String TAG_HELD_ITEM_SLOT = "HeldItemSlot";
    private static final String TAG_STATUS         = "status";
    private static final String TAG_LAST_JOB           = "lastJob";


    /**
     * The delta yaw value for looking at things.
     */
    private static final float FACING_DELTA_YAW = 10F;

    /**
     * The range in which we can hear a block break sound.
     */
    private static final double BLOCK_BREAK_SOUND_RANGE = 16.0D;

    /**
     * The range in which someone will see the particles from a block breaking.
     */
    private static final double BLOCK_BREAK_PARTICLE_RANGE = 16.0D;

    /**
     * Divide experience by a factor to ensure more levels fit in an int.
     */
    private static final double EXP_DIVIDER = 100.0;

    /**
     * Chance the citizen will rant about bad weather. 20 ticks per 60 seconds = 5 minutes.
     */
    private static final int RANT_ABOUT_WEATHER_CHANCE = 20 * 60 * 5;

    /**
     * Quantity to be moved to rotate without actually moving.
     */
    private static final double MOVE_MINIMAL = 0.001D;

    /**
     * Base max health of the citizen.
     */
    private static final double BASE_MAX_HEALTH = 20D;

    /**
     * Base movement speed of every citizen.
     */
    public static final double BASE_MOVEMENT_SPEED = 0.3D;

    /**
     * Base pathfinding range of the citizen.
     */
    private static final int    BASE_PATHFINDING_RANGE = 100;
    /**
     * Height of the citizen.
     */
    private static final double CITIZEN_HEIGHT         = 1.8D;
    /**
     * Width of the citizen.
     */
    private static final double CITIZEN_WIDTH          = 0.6D;
    /**
     * Defines how far the citizen will be rendered.
     */
    private static final double RENDER_DISTANCE_WEIGHT = 2.0D;
    /**
     * Building level at which the workers work even if it is raining.
     */
    private static final int    BONUS_BUILDING_LEVEL   = 5;
    /**
     * The speed the citizen has to rotate.
     */
    private static final double ROTATION_MOVEMENT      = 30;
    /**
     * 20 ticks or also: once a second.
     */
    private static final int    TICKS_20               = 20;
    /**
     * This times the citizen id is the personal offset of the citizen.
     */
    private static final int    OFFSET_TICK_MULTIPLIER = 7;
    /**
     * Range required for the citizen to be home.
     */
    private static final double RANGE_TO_BE_HOME       = 16;
    /**
     * The last job of the citizen.
     */
    private String lastJob                             = "";
    /**
     * If the entitiy is stuck for 2 minutes do something.
     */
    private static final int MAX_STUCK_TIME = 20*60*2;
    private static Field navigatorField;
    private final InventoryCitizen inventory;
    @NotNull
    private final Map<String, Integer> statusMessages = new HashMap<>();
    private final PathNavigate newNavigator;
    protected Status                   status  = Status.IDLE;
    private   RenderBipedCitizen.Model modelId = RenderBipedCitizen.Model.SETTLER;
    private       String           renderMetadata;
    private       ResourceLocation texture;
    private       int              colonyId;
    private int citizenId = 0;
    private int         level;
    private int         textureId;
    /**
     * Walk to proxy.
     */
    private WalkToProxy proxy;
    /**
     * Skill modifier defines how fast a citizen levels in a certain skill.
     */
    private double skillModifier = 0;
    private boolean     female;
    @Nullable
    private Colony      colony;
    @Nullable
    private CitizenData citizenData;

    /**
     * The entities current Position.
     */
    private BlockPos currentPosition = null;

    /**
     * Time the entitiy is at the same position already.
     */
    private int stuckTime = 0;


    /**
     * Citizen constructor.
     *
     * @param world the world the citizen lives in.
     */
    public EntityCitizen(final World world)
    {
        super(world);
        func_70105_a((float) CITIZEN_WIDTH, (float) CITIZEN_HEIGHT);
        this.func_110163_bv();
        this.func_174805_g(Configurations.alwaysRenderNameTag);
        this.inventory = new InventoryCitizen("Minecolonies Inventory", false, this);
        this.newNavigator = new PathNavigate(this, world);
        updateNavigatorField();
        if (world.field_72995_K)
        {
            func_184227_b(RENDER_DISTANCE_WEIGHT);
        }
        this.newNavigator.func_179693_d(true);
        this.newNavigator.func_179691_c(true);

        initTasks();
    }

    private synchronized void updateNavigatorField()
    {
        if (navigatorField == null)
        {
            final Field[] fields = EntityLiving.class.getDeclaredFields();
            for (@NotNull final Field field : fields)
            {
                if (field.getType().equals(net.minecraft.pathfinding.PathNavigate.class))
                {
                    field.setAccessible(true);
                    navigatorField = field;
                    break;
                }
            }
        }

        if (navigatorField == null)
        {
            throw new IllegalStateException("Navigator field should not be null, contact developers.");
        }

        try
        {
            navigatorField.set(this, this.newNavigator);
        }
        catch (final IllegalAccessException e)
        {
            Log.getLogger().error("Navigator error", e);
        }
    }

    /**
     * Initiates basic citizen tasks.
     */
    private void initTasks()
    {
        this.field_70714_bg.func_75776_a(0, new EntityAISwimming(this));

        if (this.getColonyJob() == null || !"com.minecolonies.coremod.job.Guard".equals(this.getColonyJob().getName()))
        {
            this.field_70714_bg.func_75776_a(1, new EntityAICitizenAvoidEntity(this, EntityMob.class, 8.0F, 0.6D, 1.6D));
        }
        this.field_70714_bg.func_75776_a(2, new EntityAIGoHome(this));
        this.field_70714_bg.func_75776_a(3, new EntityAISleep(this));
        this.field_70714_bg.func_75776_a(4, new EntityAIOpenDoor(this, true));
        this.field_70714_bg.func_75776_a(4, new EntityAIOpenFenceGate(this, true));
        this.field_70714_bg.func_75776_a(5, new EntityAIWatchClosest2(this, EntityPlayer.class, 3.0F, 1.0F));
        this.field_70714_bg.func_75776_a(6, new EntityAIWatchClosest2(this, EntityCitizen.class, 5.0F, 0.02F));
        this.field_70714_bg.func_75776_a(7, new EntityAICitizenWander(this, 0.6D));
        this.field_70714_bg.func_75776_a(8, new EntityAIWatchClosest(this, EntityLiving.class, 6.0F));

        onJobChanged(getColonyJob());
    }

    @Nullable
    public AbstractJob getColonyJob()
    {
        return citizenData == null ? null : citizenData.getJob();
    }

    /**
     * Defines job changes and state changes of the citizen.
     *
     * @param job the set job.
     */
    public void onJobChanged(@Nullable final AbstractJob job)
    {
        //  Model
        if (job == null)
        {
            switch (getLevel())
            {
                case 1:
                    modelId = RenderBipedCitizen.Model.CITIZEN;
                    break;
                case 2:
                    modelId = RenderBipedCitizen.Model.NOBLE;
                    break;
                case 3:
                    modelId = RenderBipedCitizen.Model.ARISTOCRAT;
                    break;
                default:
                    modelId = RenderBipedCitizen.Model.SETTLER;
                    break;
            }
        }
        else
        {
            modelId = job.getModel();
        }

        field_70180_af.func_187227_b(DATA_MODEL, modelId.name());
        setRenderMetadata("");


        //  AI Tasks
        @NotNull final Object[] currentTasks = this.field_70714_bg.field_75782_a.toArray();
        for (@NotNull final Object task : currentTasks)
        {
            if (((EntityAITasks.EntityAITaskEntry) task).field_75733_a instanceof AbstractEntityAIInteract)
            {
                this.field_70714_bg.func_85156_a(((EntityAITasks.EntityAITaskEntry) task).field_75733_a);
            }
        }

        if (job != null)
        {
            job.addTasks(this.field_70714_bg);
            if (field_70173_aa > 0 && getWorkBuilding() != null)
            {
                BlockPosUtil.tryMoveLivingToXYZ(this, getWorkBuilding().getLocation());
            }
        }
    }

    public int getLevel()
    {
        return level;
    }

    /**
     * On Inventory change, mark the building dirty.
     */
    public void onInventoryChanged()
    {
        final AbstractBuildingWorker building = citizenData.getWorkBuilding();
        if (building!=null)
        {
            building.markDirty();
        }
    }

    public void setRenderMetadata(final String metadata)
    {
        renderMetadata = metadata;
        field_70180_af.func_187227_b(DATA_RENDER_METADATA, renderMetadata);
        //Display some debug info always available while testing
        //tofo: remove this when in Beta!
        //Will help track down some hard to find bugs (Pathfinding etc.)
        if (citizenData != null)
        {
            if (this.getColonyJob() != null && Configurations.enableInDevelopmentFeatures)
            {
                func_96094_a(citizenData.getName() + " (" + getStatus() + ")[" + this.getColonyJob()
                                                                                       .getNameTagDescription() + "]");
            }
            else
            {
                func_96094_a(citizenData.getName());
            }
        }
    }

    /**
     * calculate this workers building.
     *
     * @return the building or null if none present.
     */
    @Nullable
    public AbstractBuildingWorker getWorkBuilding()
    {
        return (citizenData == null) ? null : citizenData.getWorkBuilding();
    }

    public Status getStatus()
    {
        return status;
    }

    public void setStatus(final Status status)
    {
        this.status = status;
    }

    /**
     * Checks if a worker is at his working site.
     * If he isn't, sets it's path to the location
     *
     * @param site  the place where he should walk to
     * @param range Range to check in
     * @return True if worker is at site, otherwise false.
     */
    public boolean isWorkerAtSiteWithMove(@NotNull final BlockPos site, final int range)
    {
        if (proxy == null)
        {
            proxy = new WalkToProxy(this);
        }
        return proxy.walkToBlock(site, range, true);
    }

    /**
     * Get the job of the citizen.
     *
     * @param type of the type.
     * @param <J>  wildcard.
     * @return the job.
     */
    @Nullable
    public <J extends AbstractJob> J getColonyJob(@NotNull final Class<J> type)
    {
        return citizenData == null ? null : citizenData.getJob(type);
    }

    /**
     * Change the citizens Rotation to look at said block.
     *
     * @param block the block he should look at.
     */
    public void faceBlock(@Nullable final BlockPos block)
    {
        if (block == null)
        {
            return;
        }

        final double xDifference = block.func_177958_n() - this.field_70165_t;
        final double zDifference = block.func_177952_p() - this.field_70161_v;
        final double yDifference = block.func_177956_o() - (this.field_70163_u + (double) this.func_70047_e());

        final double squareDifference = Math.sqrt(xDifference * xDifference + zDifference * zDifference);
        final double intendedRotationYaw = (Math.atan2(zDifference, xDifference) * 180.0D / Math.PI) - 90.0;
        final double intendedRotationPitch = -(Math.atan2(yDifference, squareDifference) * 180.0D / Math.PI);
        this.func_70101_b((float) updateRotation(this.field_70177_z, intendedRotationYaw, ROTATION_MOVEMENT),
          (float) updateRotation(this.field_70125_A, intendedRotationPitch, ROTATION_MOVEMENT));

        final double goToX = xDifference > 0 ? MOVE_MINIMAL : -MOVE_MINIMAL;
        final double goToZ = zDifference > 0 ? MOVE_MINIMAL : -MOVE_MINIMAL;

        //Have to move the entity minimally into the direction to render his new rotation.
        func_70091_d(goToX, 0, goToZ);
    }

    /**
     * Returns the new rotation degree calculated from the current and intended rotation up to a max.
     *
     * @param currentRotation  the current rotation the citizen has.
     * @param intendedRotation the wanted rotation he should have after applying this.
     * @param maxIncrement     the 'movement speed.
     * @return a rotation value he should move.
     */
    private static double updateRotation(final double currentRotation, final double intendedRotation, final double maxIncrement)
    {
        double wrappedAngle = MathHelper.func_76138_g(intendedRotation - currentRotation);

        if (wrappedAngle > maxIncrement)
        {
            wrappedAngle = maxIncrement;
        }

        if (wrappedAngle < -maxIncrement)
        {
            wrappedAngle = -maxIncrement;
        }

        return currentRotation + wrappedAngle;
    }

    /**
     * Collect exp orbs around the entity.
     */
    public void gatherXp()
    {
        for (@NotNull final EntityXPOrb orb : getXPOrbsOnGrid())
        {
            addExperience(orb.func_70526_d() / 2.0D);
            orb.func_70106_y();
        }
    }

    /**
     * Defines the area in which the citizen automatically gathers experience.
     *
     * @return a list of xp orbs around the entity.
     */
    private List<EntityXPOrb> getXPOrbsOnGrid()
    {
        @NotNull final AxisAlignedBB bb = new AxisAlignedBB(field_70165_t - 2, field_70163_u - 2, field_70161_v - 2, field_70165_t + 2, field_70163_u + 2, field_70161_v + 2);

        return field_70170_p.func_72872_a(EntityXPOrb.class, bb);
    }

    /**
     * Add experience points to citizen.
     * Increases the citizen level if he has sufficient experience.
     * This will reset the experience.
     *
     * @param xp the amount of points added.
     */
    public void addExperience(final double xp)
    {
        final double citizenHutLevel = getHomeBuilding() == null ? 0 : getHomeBuilding().getBuildingLevel();
        final double citizenHutMaxLevel = getHomeBuilding() == null ? 1 : getHomeBuilding().getMaxBuildingLevel();
        if (citizenHutLevel < citizenHutMaxLevel
              && Math.pow(2.0, citizenHutLevel + 1.0) < this.getExperienceLevel())
        {
            return;
        }

        final double maxValue = Integer.MAX_VALUE - citizenData.getExperience();
        double localXp = xp * skillModifier / EXP_DIVIDER;
        final double workBuildingLevel = getWorkBuilding() == null ? 0 : getWorkBuilding().getBuildingLevel();
        final double bonusXp = workBuildingLevel * (1 + citizenHutLevel) / Math.log(this.getExperienceLevel() + 2.0D);
        localXp = localXp * bonusXp;
        if (localXp > maxValue)
        {
            localXp = maxValue;
        }
        citizenData.addExperience(localXp);

        while (ExperienceUtils.getXPNeededForNextLevel(citizenData.getLevel()) < citizenData.getExperience())
        {
            citizenData.increaseLevel();
        }
        this.updateLevel();
        citizenData.markDirty();
    }

    private BuildingHome getHomeBuilding()
    {
        return (citizenData == null) ? null : citizenData.getHomeBuilding();
    }

    /**
     * ExperienceLevel getter.
     *
     * @return citizen ExperienceLevel value.
     */
    public int getExperienceLevel()
    {
        return citizenData.getLevel();
    }

    /**
     * Updates the level of the citizen.
     */
    private void updateLevel()
    {
        level = citizenData == null ? 0 : citizenData.getLevel();
        field_70180_af.func_187227_b(DATA_LEVEL, level);
    }

    /**
     * Entities treat being on ladders as not on ground; this breaks navigation logic.
     */
    @Override
    protected void func_184231_a(final double y, final boolean onGroundIn, final IBlockState state, final BlockPos pos)
    {
        if (!field_70122_E)
        {
            final int px = MathHelper.func_76128_c(field_70165_t);
            final int py = (int) field_70163_u;
            final int pz = MathHelper.func_76128_c(field_70161_v);

            this.field_70122_E =
              field_70170_p.func_180495_p(new BlockPos(px, py, pz)).func_177230_c().isLadder(field_70170_p.func_180495_p(new BlockPos(px, py, pz)), field_70170_p, new BlockPos(px, py, pz),
                this);
        }

        super.func_184231_a(y, onGroundIn, state, pos);
    }

    @Override
    public boolean func_70097_a(@NotNull final DamageSource damageSource, final float damage)
    {
        final Entity sourceEntity = damageSource.func_76346_g();
        if (sourceEntity instanceof EntityCitizen && ((EntityCitizen) sourceEntity).colonyId == this.colonyId)
        {
            return false;
        }

        final boolean result = super.func_70097_a(damageSource, damage);

        if (damageSource.func_82725_o() || damageSource.func_76347_k())
        {
            return result;
        }

        updateArmorDamage(damage);

        return result;
    }

    /**
     * Trigger the corresponding death achievement.
     * @param source    The damage source.
     * @param job       The job of the citizen.
     */
    public void triggerDeathAchievement(final DamageSource source, final AbstractJob job)
    {
        if (job instanceof JobMiner)
        {
            if (source == DamageSource.field_76371_c || source == DamageSource.field_76372_a || source == DamageSource.field_76370_b)
            {
                this.getColony().triggerAchievement(ModAchievements.achievementMinerDeathLava);
            }
            if (source.equals(DamageSource.field_76379_h))
            {
                this.getColony().triggerAchievement(ModAchievements.achievementMinerDeathFall);
            }
        }
        if (job instanceof JobLumberjack && source == DamageSource.field_76368_d)
        {
            this.getColony().triggerAchievement(ModAchievements.achievementLumberjackDeathTree);
        }
        if (job instanceof JobFisherman && source.func_76346_g() instanceof EntityGuardian)
        {
            this.getColony().triggerAchievement(ModAchievements.achievementFisherDeathGuardian);
        }
        if(job instanceof JobGuard && source.func_76346_g() instanceof EntityEnderman)
        {
            this.getColony().triggerAchievement(ModAchievements.achievementGuardDeathEnderman);
        }
    }

    /**
     * Called when the mob's health reaches 0.
     *
     * @param par1DamageSource the attacking entity.
     */
    @Override
    public void func_70645_a(final DamageSource par1DamageSource)
    {
        dropExperience();
        this.func_70106_y();

        if (colony != null)
        {
            triggerDeathAchievement(par1DamageSource,getColonyJob());
            if (getColonyJob() instanceof JobGuard)
            {
                LanguageHandler.sendPlayersMessage(
                  colony.getMessageEntityPlayers(),
                  "tile.blockHutTownHall.messageGuardDead",
                        citizenData.getName(), (int)field_70165_t, (int) field_70163_u, (int) field_70161_v);
            }
            else
            {
                LanguageHandler.sendPlayersMessage(
                  colony.getMessageEntityPlayers(),
                  "tile.blockHutTownHall.messageColonistDead",
                  citizenData.getName(), (int) field_70165_t, (int) field_70163_u, (int) field_70161_v);
            }
            colony.removeCitizen(getCitizenData());
        }
        super.func_70645_a(par1DamageSource);
    }

    /**
     * Drop some experience share depending on the experience and experienceLevel.
     */
    private void dropExperience()
    {
        int experience;

        if (!this.field_70170_p.field_72995_K && this.field_70718_bc > 0 && this.func_146066_aG() && this.field_70170_p.func_82736_K().func_82766_b("doMobLoot"))
        {
            experience = (int) (this.getExperiencePoints());

            while (experience > 0)
            {
                final int j = EntityXPOrb.func_70527_a(experience);
                experience -= j;
                this.field_70170_p.func_72838_d(new EntityXPOrb(this.field_70170_p, this.field_70165_t, this.field_70163_u, this.field_70161_v, j));
            }
        }

        //Spawn particle explosion of xp orbs on death
        for (int i = 0; i < 20; ++i)
        {
            final double d2 = this.field_70146_Z.nextGaussian() * 0.02D;
            final double d0 = this.field_70146_Z.nextGaussian() * 0.02D;
            final double d1 = this.field_70146_Z.nextGaussian() * 0.02D;
            this.field_70170_p.func_175688_a(EnumParticleTypes.EXPLOSION_LARGE,
              this.field_70165_t + (this.field_70146_Z.nextDouble() * this.field_70130_N * 2.0F) - (double) this.field_70130_N,
              this.field_70163_u + (this.field_70146_Z.nextDouble() * this.field_70131_O),
              this.field_70161_v + (this.field_70146_Z.nextDouble() * this.field_70130_N * 2.0F) - (double) this.field_70130_N,
              d2,
              d0,
              d1);
        }
    }

    @Nullable
    public CitizenData getCitizenData()
    {
        return citizenData;
    }

    /**
     * Get the experience points the entity currently has.
     * <p>
     *
     * @return the amount of xp this entity has
     */
    private double getExperiencePoints()
    {
        return citizenData.getExperience();
    }

    /**
     * Updates the armour damage after being hit.
     *
     * @param damage damage dealt.
     */
    private void updateArmorDamage(final double damage)
    {
        for (final ItemStack stack : this.func_184193_aE())
        {
            if (stack == null || stack.func_77973_b() == null || !(stack.func_77973_b() instanceof ItemArmor))
            {
                continue;
            }
            stack.func_77972_a((int) (damage / 2), this);

            if (stack.field_77994_a < 1)
            {
                func_184201_a(func_184640_d(stack), null);
            }
            func_184201_a(func_184640_d(stack), stack);
        }
    }

    /**
     * For the time being we don't want any childrens of our colonists.
     *
     * @param var1 the ageable entity.
     * @return the child.
     */
    @Override
    public EntityAgeable func_90011_a(final EntityAgeable var1)
    {
        return null;
    }

    /**
     * Called when a player tries to interact with a citizen.
     *
     * @param player which interacts with the citizen.
     * @return If citizen should interact or not.
     */
    @Override
    public boolean func_184645_a(@NotNull final EntityPlayer player, final EnumHand hand, @Nullable final ItemStack stack)
    {
        if (field_70170_p.field_72995_K)
        {
            final CitizenDataView citizenDataView = getCitizenDataView();
            if (citizenDataView != null)
            {
                MineColonies.proxy.showCitizenWindow(citizenDataView);
            }
        }
        return true;
    }

    @Override
    public void func_70088_a()
    {
        super.func_70088_a();
        field_70180_af.func_187214_a(DATA_COLONY_ID, colonyId);
        field_70180_af.func_187214_a(DATA_CITIZEN_ID, citizenId);
        field_70180_af.func_187214_a(DATA_TEXTURE, 0);
        field_70180_af.func_187214_a(DATA_LEVEL, 0);
        field_70180_af.func_187214_a(DATA_IS_FEMALE, 0);
        field_70180_af.func_187214_a(DATA_MODEL, RenderBipedCitizen.Model.SETTLER.name());
        field_70180_af.func_187214_a(DATA_RENDER_METADATA, "");
    }

    @Override
    public void func_70014_b(final NBTTagCompound compound)
    {
        super.func_70014_b(compound);
        compound.func_74768_a(TAG_STATUS, status.ordinal());
        if (colony != null && citizenData != null)
        {
            compound.func_74768_a(TAG_COLONY_ID, colony.getID());
            compound.func_74768_a(TAG_CITIZEN, citizenData.getId());
        }

        inventory.writeToNBT(compound);
        compound.func_74768_a(TAG_HELD_ITEM_SLOT, inventory.getHeldItemSlot());
        compound.func_74778_a(TAG_LAST_JOB, lastJob);
    }

    @Override
    public void func_70037_a(final NBTTagCompound compound)
    {
        super.func_70037_a(compound);

        status = Status.values()[compound.func_74762_e(TAG_STATUS)];
        colonyId = compound.func_74762_e(TAG_COLONY_ID);
        citizenId = compound.func_74762_e(TAG_CITIZEN);

        if (func_70613_aW())
        {
            updateColonyServer();
        }
        inventory.readFromNBT(compound);

        inventory.setHeldItem(compound.func_74762_e(TAG_HELD_ITEM_SLOT));
        lastJob = compound.func_74779_i(TAG_LAST_JOB);
    }

    /**
     * Called frequently so the entity can update its state every tick as required. For example, zombies and skeletons.
     * use this to react to sunlight and start to burn.
     */
    @Override
    public void func_70636_d()
    {
        if (field_70718_bc > 0)
        {
            citizenData.markDirty();
        }
        if (field_70170_p.field_72995_K)
        {
            updateColonyClient();
        }
        else
        {
            pickupItems();
            cleanupChatMessages();
            updateColonyServer();
            checkIfStuck();
            if (field_70170_p.func_72935_r() && !field_70170_p.func_72896_J())
            {
                SoundUtils.playRandomSound(field_70170_p, this);
            }
            else if (field_70170_p.func_72896_J() && 1 >= field_70146_Z.nextInt(RANT_ABOUT_WEATHER_CHANCE) && this.getColonyJob() != null)
            {
                SoundUtils.playSoundAtCitizenWithChance(field_70170_p, this.func_180425_c(), this.getColonyJob().getBadWeatherSound(), 1);
            }
        }

        if (func_70094_T() || func_70055_a(Material.field_151584_j))
        {
            func_70661_as().moveAwayFromXYZ(this.func_180425_c(), MOVE_AWAY_RANGE, MOVE_AWAY_SPEED);
        }

        checkHeal();
        super.func_70636_d();
    }

    private void checkIfStuck()
    {
        if(this.currentPosition == null)
        {
            this.currentPosition = this.func_180425_c();
            return;
        }

        if(this.currentPosition.equals(this.func_180425_c()) && newNavigator != null && newNavigator.getDestination() != null)
        {
            stuckTime++;
            if(stuckTime >= MAX_STUCK_TIME)
            {
                if (newNavigator.getDestination().func_177954_c(field_70165_t, field_70163_u, field_70161_v) < MOVE_AWAY_RANGE)
                {
                    stuckTime = 0;
                    return;
                }

                @Nullable final BlockPos spawnPoint =
                        Utils.scanForBlockNearPoint
                                (field_70170_p, newNavigator.getDestination(), 1, 0, 1, 2,
                                        Blocks.field_150350_a,
                                        Blocks.field_150431_aC,
                                        Blocks.field_150329_H,
                                        Blocks.field_150328_O,
                                        Blocks.field_150327_N,
                                        Blocks.field_150404_cg);

                EntityUtils.setSpawnPoint(spawnPoint, this);
                if (colony != null)
                {
                    Log.getLogger().info("Teleported stuck citizen " + this.func_70005_c_() + " from colony: " + colony.getID() + " to target location");
                }
                stuckTime = 0;
            }
        }
        else
        {
            stuckTime = 0;
            this.currentPosition = this.func_180425_c();
        }

        this.currentPosition = this.func_180425_c();
    }

    /**
     * Sets the last job of the citizen.
     * @param jobName the job he last had.
     */
    public void setLastJob(@NotNull String jobName)
    {
        this.lastJob = jobName;
    }

    /**
     * Getter for the last job.
     * @return the last job he had.
     */
    @NotNull
    public String getLastJob()
    {
        return this.lastJob;
    }

    private void updateColonyClient()
    {
        if (field_70180_af.func_187223_a())
        {
            if (colonyId == 0)
            {
                colonyId = field_70180_af.func_187225_a(DATA_COLONY_ID);
            }

            if (citizenId == 0)
            {
                citizenId = field_70180_af.func_187225_a(DATA_CITIZEN_ID);
            }

            female = field_70180_af.func_187225_a(DATA_IS_FEMALE) != 0;
            level = field_70180_af.func_187225_a(DATA_LEVEL);
            modelId = RenderBipedCitizen.Model.valueOf(field_70180_af.func_187225_a(DATA_MODEL));
            textureId = field_70180_af.func_187225_a(DATA_TEXTURE);
            renderMetadata = field_70180_af.func_187225_a(DATA_RENDER_METADATA);
            setTexture();
            field_70180_af.func_187230_e();
        }
        func_82168_bl();
    }

    /**
     * Pick up all items in a range around the citizen.
     */
    private void pickupItems()
    {
        @NotNull final List<EntityItem> retList = new ArrayList<>();
        //I know streams look better but they are flawed in type erasure
        for (final Object o : field_70170_p.func_72872_a(EntityItem.class, func_174813_aQ().func_72314_b(2.0F, 0.0F, 2.0F)))
        {
            if (o instanceof EntityItem)
            {
                retList.add((EntityItem) o);
            }
        }

        retList.stream()
          .filter(item -> item != null)
          .filter(item -> !item.field_70128_L)
          .filter(item -> func_98052_bS())
          .forEach(this::tryPickupEntityItem);
    }

    private void cleanupChatMessages()
    {
        //Only check if there are messages and once a second
        if (statusMessages.size() > 0 && field_70173_aa % TICKS_20 == 0)
        {
            @NotNull final Iterator<Map.Entry<String, Integer>> it = statusMessages.entrySet().iterator();
            while (it.hasNext())
            {
                if (field_70173_aa - it.next().getValue() > TICKS_20 * Configurations.chatFrequency)
                {
                    it.remove();
                }
            }
        }
    }

    /**
     * Checks the citizens health status and heals the citizen if necessary.
     */
    private void checkHeal()
    {
        if (citizenData != null && getOffsetTicks() % HEAL_CITIZENS_AFTER == 0 && func_110143_aJ() < func_110138_aP())
        {
            func_70691_i(1);
            citizenData.markDirty();
        }
    }

    /**
     * Sets the textures of all citizens and distinguishes between male and female.
     */
    private void setTexture()
    {
        if (!field_70170_p.field_72995_K)
        {
            return;
        }

        final RenderBipedCitizen.Model model = getModelID();

        String textureBase = "textures/entity/" + model.textureBase;
        textureBase += female ? "Female" : "Male";

        final int moddedTextureId = (textureId % model.numTextures) + 1;
        texture = new ResourceLocation(Constants.MOD_ID, textureBase + moddedTextureId + renderMetadata + ".png");
    }

    public int getOffsetTicks()
    {
        return this.field_70173_aa + OFFSET_TICK_MULTIPLIER * this.func_145782_y();
    }

    public RenderBipedCitizen.Model getModelID()
    {
        return modelId;
    }

    /**
     * Server-specific update for the EntityCitizen.
     */
    public void updateColonyServer()
    {
        if (colonyId == 0)
        {
            func_70106_y();
            return;
        }

        if (colony == null)
        {
            handleNullColony();
        }
    }

    /**
     * Handles extreme cases like colony or citizen is null.
     */
    private void handleNullColony()
    {
        final Colony c = ColonyManager.getColony(colonyId);

        if (c == null)
        {
            Log.getLogger().warn(String.format("EntityCitizen '%s' unable to find Colony #%d", func_110124_au(), colonyId));
            func_70106_y();
            return;
        }

        final CitizenData data = c.getCitizen(citizenId);
        if (data == null)
        {
            //  Citizen does not exist in the Colony
            Log.getLogger().warn(String.format("EntityCitizen '%s' attempting to register with Colony #%d as Citizen %d, but not known to colony",
              func_110124_au(),
              colonyId,
              citizenId));
            func_70106_y();
            return;
        }

        @Nullable final EntityCitizen existingCitizen = data.getCitizenEntity();
        if (existingCitizen != null && existingCitizen != this)
        {
            // This Citizen already has a different Entity registered to it
            handleExistingCitizen(data, existingCitizen);
            return;
        }

        setColony(c, data);
    }

    private void handleExistingCitizen(@NotNull final CitizenData data, @NotNull final EntityCitizen existingCitizen)
    {
        Log.getLogger().warn(String.format("EntityCitizen '%s' attempting to register with Colony #%d as Citizen #%d, but already have a citizen ('%s')",
          func_110124_au(),
          colonyId,
          citizenId,
          existingCitizen.func_110124_au()));
        if (existingCitizen.func_110124_au().equals(this.func_110124_au()))
        {
            data.setCitizenEntity(this);
        }
        else
        {
            func_70106_y();
        }
    }

    /**
     * Assigns a citizen to a colony.
     *
     * @param c    the colony.
     * @param data the data of the new citizen.
     */
    public void setColony(@Nullable final Colony c, @Nullable final CitizenData data)
    {
        if (c == null)
        {
            colony = null;
            colonyId = 0;
            citizenId = 0;
            citizenData = null;
            func_70106_y();
            return;
        }

        colony = c;
        colonyId = colony.getID();
        citizenId = data.getId();
        citizenData = data;

        func_96094_a(citizenData.getName());

        female = citizenData.isFemale();
        textureId = citizenData.getTextureId();

        field_70180_af.func_187227_b(DATA_COLONY_ID, colonyId);
        field_70180_af.func_187227_b(DATA_CITIZEN_ID, citizenId);
        field_70180_af.func_187227_b(DATA_IS_FEMALE, female ? 1 : 0);
        field_70180_af.func_187227_b(DATA_TEXTURE, textureId);
        updateLevel();

        citizenData.setCitizenEntity(this);

        onJobChanged(getColonyJob());
    }

    /**
     * Getter of the dataview, the clientside representation of the citizen.
     *
     * @return the view.
     */
    private CitizenDataView getCitizenDataView()
    {
        if (colonyId != 0 && citizenId != 0)
        {
            final ColonyView colonyView = ColonyManager.getColonyView(colonyId);
            if (colonyView != null)
            {
                return colonyView.getCitizen(citizenId);
            }
        }

        return null;
    }

    /**
     * Getter of the citizens random object.
     *
     * @return random object.
     */
    public Random getRandom()
    {
        return field_70146_Z;
    }

    /**
     * Applies attributes like health, charisma etc to the citizens.
     */
    @Override
    protected void func_110147_ax()
    {
        super.func_110147_ax();

        func_110148_a(SharedMonsterAttributes.field_111267_a).func_111128_a(BASE_MAX_HEALTH);
        func_110148_a(SharedMonsterAttributes.field_111263_d).func_111128_a(BASE_MOVEMENT_SPEED);

        //path finding search range
        func_110148_a(SharedMonsterAttributes.field_111265_b).func_111128_a(BASE_PATHFINDING_RANGE);
    }

    @NotNull
    @Override
    public PathNavigate func_70661_as()
    {
        return newNavigator;
    }

    /**
     * Drop the equipment for this entity.
     */
    @Override
    protected void func_82160_b(final boolean par1, final int par2)
    {
        //Drop actual inventory
        for (int i = 0; i < inventory.func_70302_i_(); i++)
        {
            final ItemStack itemstack = inventory.func_70301_a(i);
            if (itemstack != null && itemstack.field_77994_a > 0)
            {
                entityDropItem(itemstack);
            }
        }
    }

    /**
     * Returns false if the newer Entity AI code should be run.
     */
    @Override
    public boolean func_175446_cd()
    {
        return false;
    }

    /**
     * Handles the dropping of items from the entity.
     *
     * @param itemstack to drop.
     * @return the dropped item.
     */
    private EntityItem entityDropItem(@NotNull final ItemStack itemstack)
    {
        return func_70099_a(itemstack, 0.0F);
    }

    /**
     * Getter of the resource location of the texture.
     *
     * @return location of the texture.
     */
    public ResourceLocation getTexture()
    {
        return texture;
    }

    /**
     * Getter which checks if the citizen is female.
     *
     * @return true if female.
     */
    public boolean isFemale()
    {
        return female;
    }

    /**
     * Clears the colony of the citizen.
     */
    public void clearColony()
    {
        setColony(null, null);
    }

    public boolean isAtHome()
    {
        @Nullable final BlockPos homePosition = func_180486_cf();
        return homePosition != null && homePosition.func_177954_c((int) Math.floor(field_70165_t), (int) field_70163_u, (int) Math.floor(field_70161_v)) <= RANGE_TO_BE_HOME;
    }

    /**
     * Returns the home position of each citizen (His house or town hall).
     *
     * @return location
     */
    @Nullable
    @Override
    public BlockPos func_180486_cf()
    {
        @Nullable final BuildingHome homeBuilding = getHomeBuilding();
        if (homeBuilding != null)
        {
            return homeBuilding.getLocation();
        }
        else if (getColony() != null && getColony().getTownHall() != null)
        {
            return getColony().getTownHall().getLocation();
        }

        return null;
    }

    @Nullable
    public Colony getColony()
    {
        return colony;
    }

    public boolean isInventoryFull()
    {
        return InventoryUtils.isInventoryFull(getInventoryCitizen());
    }

    /**
     * Return this citizens inventory.
     *
     * @return the inventory this citizen has.
     */
    @NotNull
    public InventoryCitizen getInventoryCitizen()
    {
        return inventory;
    }

    @NotNull
    public DesiredActivity getDesiredActivity()
    {
        if (this.getColonyJob() instanceof JobGuard)
        {
            return DesiredActivity.WORK;
        }

        if (!field_70170_p.func_72935_r())
        {
            return DesiredActivity.SLEEP;
        }
        else if (field_70170_p.func_72896_J() && !shouldWorkWhileRaining())
        {
            return DesiredActivity.IDLE;
        }
        else
        {
            if(this.func_70661_as() != null && (this.func_70661_as().func_75505_d() != null && this.func_70661_as().func_75505_d().func_75874_d() == 0))
            {
                this.func_70661_as().func_75499_g();
            }
            return DesiredActivity.WORK;
        }
    }

    /**
     * Checks if the citizen should work even when it rains.
     *
     * @return true if his building level is bigger than 5.
     */
    private boolean shouldWorkWhileRaining()
    {
        return this.getWorkBuilding() != null && (this.getWorkBuilding().getBuildingLevel() >= BONUS_BUILDING_LEVEL);
    }

    /**
     * We override this method and execute no code to avoid citizens travelling to the nether.
     *
     * @param dimensionIn dimension to travel to.
     */
    @Override
    @Nullable
    public Entity func_184204_a(final int dimensionIn)
    {
        return null;
    }

    @NotNull
    @Override
    public BlockPos func_180425_c()
    {
        return new BlockPos(field_70165_t, field_70163_u, field_70161_v);
    }

    /**
     * Returns the first slot in the inventory with a specific item.
     *
     * @param targetItem the item.
     * @param itemDamage the damage value
     * @return the slot.
     */
    public int findFirstSlotInInventoryWith(final Item targetItem, int itemDamage)
    {
        return InventoryUtils.findFirstSlotInInventoryWith(getInventoryCitizen(), targetItem, itemDamage);
    }

    /**
     * Returns the first slot in the inventory with a specific block.
     *
     * @param block the block.
     * @param itemDamage the damage value
     * @return the slot.
     */
    public int findFirstSlotInInventoryWith(final Block block, int itemDamage)
    {
        return InventoryUtils.findFirstSlotInInventoryWith(getInventoryCitizen(), block, itemDamage);
    }

    /**
     * Returns the amount of a certain block in the inventory.
     *
     * @param block the block.
     * @param itemDamage the damage value
     * @return the quantity.
     */
    public int getItemCountInInventory(final Block block, int itemDamage)
    {
        return InventoryUtils.getItemCountInInventory(getInventoryCitizen(), block, itemDamage);
    }

    /**
     * Returns the amount of a certain item in the inventory.
     *
     * @param targetItem the block.
     * @param itemDamage the damage value.
     * @return the quantity.
     */
    public int getItemCountInInventory(final Item targetItem, int itemDamage)
    {
        return InventoryUtils.getItemCountInInventory(getInventoryCitizen(), targetItem, itemDamage);
    }

    /**
     * Checks if citizen has a certain block in the inventory.
     *
     * @param block the block.
     * @param itemDamage the damage value
     * @return true if so.
     */
    public boolean hasItemInInventory(final Block block, int itemDamage)
    {
        return InventoryUtils.hasitemInInventory(getInventoryCitizen(), block, itemDamage);
    }

    /**
     * Checks if citizen has a certain item in the inventory.
     *
     * @param item the item.
     * @param itemDamage the damage value
     * @return true if so.
     */
    public boolean hasItemInInventory(final Item item, int itemDamage)
    {
        return InventoryUtils.hasitemInInventory(getInventoryCitizen(), item, itemDamage);
    }

    /**
     * Citizen will try to pick up a certain item.
     *
     * @param entityItem the item he wants to pickup.
     */
    private void tryPickupEntityItem(@NotNull final EntityItem entityItem)
    {
        if (!this.field_70170_p.field_72995_K)
        {
            if (entityItem.func_174874_s())
            {
                return;
            }

            final ItemStack itemStack = entityItem.func_92059_d();

            final int i = itemStack.field_77994_a;
            if (i <= 0 || InventoryUtils.addItemStackToInventory(this.getInventoryCitizen(), itemStack))
            {
                this.field_70170_p.func_184133_a((EntityPlayer) null,
                  this.func_180425_c(),
                  SoundEvents.field_187638_cR,
                  SoundCategory.AMBIENT,
                  0.2F,
                  (float) ((this.field_70146_Z.nextGaussian() * 0.7D + 1.0D) * 2.0D));
                this.func_71001_a(this, i);

                if (itemStack.field_77994_a <= 0)
                {
                    entityItem.func_70106_y();
                }
            }
        }
    }

    /**
     * Removes the currently held item.
     */
    public void removeHeldItem()
    {
        func_184201_a(EntityEquipmentSlot.MAINHAND, null);
    }

    /**
     * Sets the currently held item.
     *
     * @param slot from the inventory slot.
     */
    public void setHeldItem(final int slot)
    {
        inventory.setHeldItem(slot);
        func_184201_a(EntityEquipmentSlot.MAINHAND, inventory.func_70301_a(slot));
    }

    /**
     * Swing entity arm, create sound and particle effects.
     * <p>
     * Will not break the block.
     *
     * @param blockPos Block position.
     */
    public void hitBlockWithToolInHand(@Nullable final BlockPos blockPos)
    {
        if (blockPos == null)
        {
            return;
        }
        hitBlockWithToolInHand(blockPos, false);
    }

    /**
     * Swing entity arm, create sound and particle effects.
     * <p>
     * If breakBlock is true then it will break the block (different sound and particles),
     * and damage the tool in the citizens hand.
     *
     * @param blockPos   Block position.
     * @param breakBlock if we want to break this block.
     */
    private void hitBlockWithToolInHand(@Nullable final BlockPos blockPos, final boolean breakBlock)
    {
        if (blockPos == null)
        {
            return;
        }

        this.func_70671_ap().func_75650_a(blockPos.func_177958_n(), blockPos.func_177956_o(), blockPos.func_177952_p(), FACING_DELTA_YAW, func_70646_bf());

        this.func_184609_a(this.func_184600_cs());

        final IBlockState blockState = field_70170_p.func_180495_p(blockPos);
        final Block block = blockState.func_177230_c();
        if (breakBlock)
        {
            if (!field_70170_p.field_72995_K)
            {
                MineColonies.getNetwork().sendToAllAround(
                  new BlockParticleEffectMessage(blockPos, field_70170_p.func_180495_p(blockPos), BlockParticleEffectMessage.BREAK_BLOCK),
                  new NetworkRegistry.TargetPoint(field_70170_p.field_73011_w.getDimension(), blockPos.func_177958_n(), blockPos.func_177956_o(), blockPos.func_177952_p(), BLOCK_BREAK_SOUND_RANGE));
            }
            field_70170_p.func_184133_a(null,
              blockPos,
              block.getSoundType(blockState, field_70170_p, blockPos, this).func_185845_c(),
              SoundCategory.BLOCKS,
              block.getSoundType(blockState, field_70170_p, blockPos, this).func_185843_a(),
              block.getSoundType(blockState, field_70170_p, blockPos, this).func_185847_b());
            field_70170_p.func_175698_g(blockPos);

            damageItemInHand(1);
        }
        else
        {
            //todo: might remove this
            if (!field_70170_p.field_72995_K)
            {
                MineColonies.getNetwork().sendToAllAround(
                  //todo: correct side
                  new BlockParticleEffectMessage(blockPos, field_70170_p.func_180495_p(blockPos), 1),
                  new NetworkRegistry.TargetPoint(field_70170_p.field_73011_w.getDimension(), blockPos.func_177958_n(), blockPos.func_177956_o(), blockPos.func_177952_p(), BLOCK_BREAK_PARTICLE_RANGE));
            }
            field_70170_p.func_184133_a((EntityPlayer) null,
              blockPos,
              block.getSoundType(blockState, field_70170_p, blockPos, this).func_185845_c(),
              SoundCategory.BLOCKS,
              block.getSoundType(blockState, field_70170_p, blockPos, this).func_185843_a(),
              block.getSoundType(blockState, field_70170_p, blockPos, this).func_185847_b());
        }
    }

    /**
     * Damage the current held item.
     *
     * @param damage amount of damage.
     */
    public void damageItemInHand(final int damage)
    {
        final ItemStack heldItem = inventory.getHeldItemMainhand();
        //If we hit with bare hands, ignore
        if (heldItem == null)
        {
            return;
        }
        heldItem.func_77972_a(damage, this);

        //check if tool breaks
        if (heldItem.field_77994_a < 1)
        {
            getInventoryCitizen().func_70299_a(getInventoryCitizen().getHeldItemSlot(), null);
            this.func_184201_a(EntityEquipmentSlot.MAINHAND, null);
        }
    }

    /**
     * Swing entity arm, create sound and particle effects.
     * <p>
     * This will break the block (different sound and particles),
     * and damage the tool in the citizens hand.
     *
     * @param blockPos Block position.
     */
    public void breakBlockWithToolInHand(@Nullable final BlockPos blockPos)
    {
        if (blockPos == null)
        {
            return;
        }
        hitBlockWithToolInHand(blockPos, true);
    }

    /**
     * Sends a localized message from the citizen containing a language string with a key and arguments.
     *
     * @param key  the key to retrieve the string.
     * @param args additional arguments.
     */
    public void sendLocalizedChat(final String key, final Object... args)
    {
        sendChat(key, args);
    }

    /**
     * Sends a chat string close to the citizen.
     *
     * @param msg the message string.
     */
    private void sendChat(final String key, @Nullable final Object... msg)
    {
        if (msg == null || statusMessages.containsKey(key))
        {
            return;
        }

        final TextComponentTranslation requiredItem;

        if(msg.length == 0)
        {
            requiredItem = new TextComponentTranslation(key);
        }
        else
        {
            statusMessages.put(key + msg[0], field_70173_aa);
            requiredItem = new TextComponentTranslation(key, msg);
        }

        final TextComponentString citizenDescription = new TextComponentString(" ");
        citizenDescription.func_150258_a(this.func_95999_t()).func_150258_a(": ");
        final TextComponentString colonyDescription = new TextComponentString(" at " + this.getColony().getName() + ":");

        LanguageHandler.sendPlayersMessage(colony.getMessageEntityPlayers(),  this.getColonyJob().getName(), colonyDescription, citizenDescription, requiredItem);
    }

    /**
     * Intelligence getter.
     *
     * @return citizen intelligence value.
     */
    public int getIntelligence()
    {
        return citizenData.getIntelligence();
    }

    /**
     * Charisma getter.
     *
     * @return citizen Charisma value.
     */
    public int getCharisma()
    {
        return citizenData.getCharisma();
    }

    /**
     * Strength getter.
     *
     * @return citizen Strength value.
     */
    public int getStrength()
    {
        return citizenData.getStrength();
    }

    /**
     * Endurance getter.
     *
     * @return citizen Endurance value.
     */
    public int getEndurance()
    {
        return citizenData.getEndurance();
    }

    /**
     * Dexterity getter.
     *
     * @return citizen Dexterity value.
     */
    public int getDexterity()
    {
        return citizenData.getDexterity();
    }

    /**
     * Set the skill modifier which defines how fast a citizen levels in a certain skill.
     *
     * @param modifier input modifier.
     */
    public void setSkillModifier(final int modifier)
    {
        skillModifier = modifier;
    }

    /**
     * Called when the citizen wakes up.
     */
    public void onWakeUp()
    {
        if (this.getWorkBuilding() instanceof BuildingFarmer)
        {
            ((BuildingFarmer) this.getWorkBuilding()).resetFields();
        }
    }

    /**
     * Enum describing the citizens activity.
     */
    public enum DesiredActivity
    {
        SLEEP,
        IDLE,
        WORK
    }

    /**
     * Used for chat messages, sounds, and other need based interactions.
     * Created: June 20, 2014
     *
     * @author Colton
     */
    public enum Status
    {
        IDLE,
        SLEEPING,
        WORKING,
        GETTING_ITEMS,
        NEED_ASSISTANCE,
        PATHFINDING_ERROR
    }
}
