package com.minecolonies.coremod.entity.ai.basic;

import com.minecolonies.coremod.blocks.AbstractBlockHut;
import com.minecolonies.coremod.blocks.ModBlocks;
import com.minecolonies.coremod.colony.buildings.AbstractBuilding;
import com.minecolonies.coremod.colony.buildings.AbstractBuildingWorker;
import com.minecolonies.coremod.colony.buildings.BuildingBuilder;
import com.minecolonies.coremod.colony.buildings.BuildingMiner;
import com.minecolonies.coremod.colony.buildings.utils.BuildingBuilderResource;
import com.minecolonies.coremod.colony.jobs.AbstractJob;
import com.minecolonies.coremod.colony.jobs.AbstractJobStructure;
import com.minecolonies.coremod.colony.jobs.JobBuilder;
import com.minecolonies.coremod.colony.jobs.JobMiner;
import com.minecolonies.coremod.colony.workorders.WorkOrderBuild;
import com.minecolonies.coremod.colony.workorders.WorkOrderBuildDecoration;
import com.minecolonies.coremod.configuration.Configurations;
import com.minecolonies.coremod.entity.ai.citizen.miner.Level;
import com.minecolonies.coremod.entity.ai.citizen.miner.Node;
import com.minecolonies.coremod.entity.ai.util.AIState;
import com.minecolonies.coremod.entity.ai.util.AITarget;
import com.minecolonies.coremod.entity.ai.util.Structure;
import com.minecolonies.coremod.util.*;
import net.minecraft.block.*;
import net.minecraft.block.state.IBlockState;
import net.minecraft.entity.Entity;
import net.minecraft.entity.EntityList;
import net.minecraft.entity.item.EntityArmorStand;
import net.minecraft.entity.item.EntityItemFrame;
import net.minecraft.init.Blocks;
import net.minecraft.init.Items;
import net.minecraft.inventory.EntityEquipmentSlot;
import net.minecraft.item.ItemDoor;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.tileentity.TileEntityFlowerPot;
import net.minecraft.tileentity.TileEntityLockable;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.Mirror;
import net.minecraft.util.math.AxisAlignedBB;
import net.minecraft.util.math.BlockPos;
import net.minecraft.util.math.MathHelper;
import net.minecraft.util.math.RayTraceResult;
import net.minecraft.world.World;
import net.minecraft.world.gen.structure.template.Template;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.awt.geom.Point2D;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;
import java.util.function.Function;
import java.util.function.Supplier;

import static com.minecolonies.coremod.entity.ai.util.AIState.*;

/**
 * This base ai class is used by ai's who need to build entire structures.
 * These structures have to be supplied as schematics files.
 * <p>
 * Once an ai starts building a structure, control over it is only given back once that is done.
 * <p>
 * If the ai resets, the structure is gone,
 * so just restart building and no progress will be reset.
 *
 * @param <J> the job type this AI has to do.
 */
public abstract class AbstractEntityAIStructure<J extends AbstractJob> extends AbstractEntityAIInteract<J>
{
    /**
     * Lead the miner to the other side of the shaft.
     */
    private static final int OTHER_SIDE_OF_SHAFT = 6;
    /**
     * Amount of xp the builder gains each building (Will increase by attribute modifiers additionally).
     */
    private static final double XP_EACH_BUILDING              = 2.5;
    /**
     * Speed the builder should run away when he castles himself in.
     */
    private static final double RUN_AWAY_SPEED                = 4.1D;
    /**
     * The minimum range to keep from the current building place.
     */
    private static final int    MIN_ADDITIONAL_RANGE_TO_BUILD = 3;
    /**
     * The maximum range to keep from the current building place.
     */
    private static final int    MAX_ADDITIONAL_RANGE_TO_BUILD = 25;
    /**
     * The amount of ticks to wait when not needing any tools to break blocks.
     */
    private static final int    UNLIMITED_RESOURCES_TIMEOUT   = 5;
    /**
     * The current structure task to be build.
     */
    private Structure currentStructure;
    /**
     * Position where the Builders constructs from.
     */
    private BlockPos  workFrom;
    /**
     * The standard range the builder should reach until his target.
     */
    private static final int STANDARD_WORKING_RANGE = 5;
    /**
     * The minimum range the builder has to reach in order to construct or clear.
     */
    private static final int MIN_WORKING_RANGE      = 12;

    private int rotation = 0;

    /**
     * String which shows if something is a waypoint.
     */
    private static final CharSequence WAYPOINT_STRING = "waypoint";

    /**
     * Creates this ai base class and set's up important things.
     * <p>
     * Always use this constructor!
     *
     * @param job the job class of the ai using this base class.
     */
    protected AbstractEntityAIStructure(@NotNull final J job)
    {
        super(job);
        this.registerTargets(

                /**
                 * Check if tasks should be executed.
                 */
                new AITarget(this::checkIfCanceled, IDLE),
                /**
                 * Select the appropriate State to do next.
                 */
                new AITarget(START_BUILDING, this::startBuilding),
                /**
                 * Check if we have to build something.
                 */
                new AITarget(IDLE, this::isThereAStructureToBuild, () -> AIState.START_BUILDING),
                /**
                 * Clear out the building area.
                 */
                new AITarget(CLEAR_STEP, generateStructureGenerator(this::clearStep, AIState.BUILDING_STEP)),
                /**
                 * Build the structure and foundation of the building.
                 */
                new AITarget(BUILDING_STEP, generateStructureGenerator(this::structureStep, AIState.DECORATION_STEP)),
                /**
                 * Decorate the AbstractBuilding with torches etc.
                 */
                new AITarget(DECORATION_STEP, generateStructureGenerator(this::decorationStep, AIState.SPAWN_STEP)),
                /**
                 * Spawn entities on the structure.
                 */
                new AITarget(SPAWN_STEP, generateStructureGenerator(this::spawnEntity, AIState.COMPLETE_BUILD)),
                /**
                 * Finalize the building and give back control to the ai.
                 */
                new AITarget(COMPLETE_BUILD, this::completeBuild)
        );
    }

    private AIState completeBuild()
    {
        if (job instanceof AbstractJobStructure)
        {
            if (((AbstractJobStructure) job).getStructure() == null && job instanceof JobBuilder && ((JobBuilder) job).hasWorkOrder())
            {
                //fix for bad structures
                ((JobBuilder) job).complete();
            }

            if (job instanceof JobBuilder && ((JobBuilder) job).getStructure() != null)
            {
                final String structureName = ((AbstractJobStructure) job).getStructure().getName();
                LanguageHandler.sendPlayersMessage(worker.getColony().getMessageEntityPlayers(),
                        "entity.builder.messageBuildComplete",
                        structureName);


                final WorkOrderBuild wo = ((JobBuilder) job).getWorkOrder();
                if (wo == null)
                {
                    Log.getLogger().error(String.format("Builder (%d:%d) ERROR - Finished, but missing work order(%d)",
                            worker.getColony().getID(),
                            worker.getCitizenData().getId(),
                            ((JobBuilder) job).getWorkOrderId()));
                }
                else
                {
                    if (wo instanceof WorkOrderBuildDecoration)
                    {
                        if (structureName.contains(WAYPOINT_STRING))
                        {
                            worker.getColony().addWayPoint(wo.getBuildingLocation(), world.func_180495_p(wo.getBuildingLocation()));
                        }
                    }
                    else
                    {
                        final AbstractBuilding building = job.getColony().getBuilding(wo.getBuildingLocation());
                        if (building == null)
                        {
                            Log.getLogger().error(String.format("Builder (%d:%d) ERROR - Finished, but missing building(%s)",
                                    worker.getColony().getID(),
                                    worker.getCitizenData().getId(),
                                    wo.getBuildingLocation()));
                        }
                        else
                        {
                            building.setBuildingLevel(wo.getUpgradeLevel());
                        }
                    }
                    ((JobBuilder) job).complete();
                }

                final AbstractBuilding workerBuilding = getOwnBuilding();
                if (workerBuilding instanceof BuildingBuilder)
                {
                    ((BuildingBuilder) workerBuilding).resetNeededResources();
                }
                func_75251_c();
            }
            else if (job instanceof JobMiner)
            {
                final BuildingMiner minerBuilding = (BuildingMiner) getOwnBuilding();
                //If shaft isn't cleared we're in shaft clearing mode.
                if (minerBuilding.clearedShaft)
                {
                    minerBuilding.getCurrentLevel().closeNextNode(rotation);
                }
                else
                {
                    @NotNull final Level currentLevel = new Level(minerBuilding, ((JobMiner) job).getStructure().getPosition().func_177956_o());
                    minerBuilding.addLevel(currentLevel);
                    minerBuilding.setCurrentLevel(minerBuilding.getNumberOfLevels());
                    minerBuilding.resetStartingLevelShaft();
                }
                //Send out update to client
                getOwnBuilding().markDirty();

                ((JobMiner) job).setStructure(null);
            }
            worker.addExperience(XP_EACH_BUILDING);
        }

        workFrom = null;
        currentStructure = null;

        return AIState.IDLE;
    }

    private Boolean decorationStep(final Structure.StructureBlock structureBlock)
    {
        if (!BlockUtils.shouldNeverBeMessedWith(structureBlock.worldBlock))
        {
            //Fill workFrom with the position from where the builder should build.
            //also ensure we are at that position.
            if (!walkToConstructionSite())
            {
                return false;
            }

            if (structureBlock.block == null
                    || structureBlock.doesStructureBlockEqualWorldBlock()
                    || structureBlock.metadata.func_185904_a().func_76220_a()
                    || (structureBlock.block instanceof BlockBed && structureBlock.metadata.func_177229_b(BlockBed.field_176472_a).equals(BlockBed.EnumPartType.HEAD))
                    || (structureBlock.block instanceof BlockDoor && structureBlock.metadata.func_177229_b(BlockDoor.field_176523_O).equals(BlockDoor.EnumDoorHalf.UPPER)))
            {
                //findNextBlock count was reached and we can ignore this block
                return true;
            }

            worker.faceBlock(structureBlock.blockPosition);

            @Nullable final Block block = structureBlock.block;

            //should never happen
            if (block == null)
            {
                @NotNull final BlockPos local = structureBlock.blockPosition;
                Log.getLogger().error(String.format("StructureProxy has null block at %s - local(%s)", currentStructure.getCurrentBlockPosition(), local));
                return true;
            }

            @Nullable final IBlockState blockState = structureBlock.metadata;
            //We need to deal with materials
            if (!Configurations.builderInfiniteResources
                    && !handleMaterials(block, blockState))
            {
                return false;
            }

            placeBlockAt(block, blockState, structureBlock.blockPosition);
        }
        return true;
    }

    private Boolean structureStep(final Structure.StructureBlock structureBlock)
    {
        if (!BlockUtils.shouldNeverBeMessedWith(structureBlock.worldBlock))
        {
            //Fill workFrom with the position from where the builder should build.
            //also ensure we are at that position.
            if (!walkToConstructionSite())
            {
                return false;
            }

            if (structureBlock.block == null
                    || structureBlock.doesStructureBlockEqualWorldBlock()
                    || (!structureBlock.metadata.func_185904_a().func_76220_a() && structureBlock.block != Blocks.field_150350_a)
                    || (structureBlock.block instanceof BlockBed && structureBlock.metadata.func_177229_b(BlockBed.field_176472_a).equals(BlockBed.EnumPartType.HEAD))
                    || (structureBlock.block instanceof BlockDoor && structureBlock.metadata.func_177229_b(BlockDoor.field_176523_O).equals(BlockDoor.EnumDoorHalf.UPPER)))
            {
                //findNextBlock count was reached and we can ignore this block
                return true;
            }

            @Nullable Block block = structureBlock.block;
            @Nullable IBlockState blockState = structureBlock.metadata;
            if (structureBlock.block == ModBlocks.blockSolidSubstitution)
            {
                if (!(job instanceof JobMiner && structureBlock.worldBlock instanceof BlockOre)
                        && structureBlock.worldMetadata.func_185904_a().func_76220_a())
                {
                    return true;
                }
                blockState = getSolidSubstitution(structureBlock.blockPosition);
                block = blockState.func_177230_c();
            }

            worker.faceBlock(structureBlock.blockPosition);

            //should never happen
            if (block == null)
            {
                @NotNull final BlockPos local = structureBlock.blockPosition;
                Log.getLogger().error(String.format("StructureProxy has null block at %s - local(%s)", currentStructure.getCurrentBlockPosition(), local));
                return true;
            }

            //We need to deal with materials
            if (!Configurations.builderInfiniteResources
                    && !handleMaterials(block, blockState))
            {
                return false;
            }

            placeBlockAt(block, blockState, structureBlock.blockPosition);
        }
        return true;
    }

    /**
     * Generate a function that will iterate over a structure.
     * <p>
     * It will pass the current block (with all infos) to the evaluation function.
     *
     * @param evaluationFunction the function to be called each block.
     * @param nextState          the next state to change to once done iterating.
     * @return the new state this AI will be in after one pass.
     */
    private Supplier<AIState> generateStructureGenerator(@NotNull final Function<Structure.StructureBlock, Boolean> evaluationFunction, @NotNull final AIState nextState)
    {
        //do not replace with method reference, this one stays the same on changing reference for currentStructure
        //URGENT: DO NOT REPLACE FOR ANY MEANS THIS WILL CRASH THE GAME.
        @NotNull final Supplier<Structure.StructureBlock> getCurrentBlock = () -> currentStructure.getCurrentBlock();
        @NotNull final Supplier<Structure.Result> advanceBlock = () -> currentStructure.advanceBlock();

        return () ->
        {
            final Structure.StructureBlock currentBlock = getCurrentBlock.get();
            /*
            check if we have not found a block (when block == null
            if we have a block, apply the eval function
            (which changes stuff, so only execute on valid block!)
            */
            if (currentBlock.block == null
                    || evaluationFunction.apply(currentBlock))
            {
                final Structure.Result result = advanceBlock.get();
                if (result == Structure.Result.AT_END)
                {
                    switchStage(nextState);
                    return nextState;
                }
                if (result == Structure.Result.CONFIG_LIMIT)
                {
                    return getState();
                }
            }
            return getState();
        };
    }

    /**
     * Switches the structures stage after the current one has been completed.
     */
    private void switchStage(AIState state)
    {
        if (state.equals(AIState.BUILDING_STEP))
        {
            currentStructure.setStage(Structure.Stage.BUILD);
        }
        else if (state.equals(AIState.DECORATION_STEP))
        {
            currentStructure.setStage(Structure.Stage.DECORATE);
        }
        else if (state.equals(AIState.SPAWN_STEP))
        {
            currentStructure.setStage(Structure.Stage.SPAWN);
        }
        else if (state.equals(AIState.COMPLETE_BUILD))
        {
            currentStructure.setStage(Structure.Stage.COMPLETE);
        }
    }

    /**
     * Load the structure, special builder use with workOrders.
     * Extracts data from workOrder and hands it to generic loading.
     */
    public void loadStructure()
    {
        WorkOrderBuild workOrder = null;
        if (job instanceof JobBuilder)
        {
            workOrder = ((JobBuilder) job).getWorkOrder();
        }

        if (workOrder == null)
        {
            return;
        }

        final BlockPos pos = workOrder.getBuildingLocation();
        if (!(workOrder instanceof WorkOrderBuildDecoration) && worker.getColony().getBuilding(pos) == null)
        {
            Log.getLogger().warn("AbstractBuilding does not exist - removing build request");
            worker.getColony().getWorkManager().removeWorkOrder(workOrder);
            return;
        }

        int tempRotation = 0;
        if (workOrder.getRotation() == 0 && !(workOrder instanceof WorkOrderBuildDecoration))
        {
            final IBlockState blockState = world.func_180495_p(pos);
            if (blockState.func_177230_c() instanceof AbstractBlockHut)
            {
                tempRotation = BlockUtils.getRotationFromFacing(blockState.func_177229_b(AbstractBlockHut.FACING));
            }
        }
        else
        {
            tempRotation = workOrder.getRotation();
        }

        loadStructure(workOrder.getStructureName(), tempRotation, pos, workOrder.isMirrored());

        workOrder.setCleared(false);
        workOrder.setRequested(false);

        //We need to deal with materials
        requestMaterialsIfRequired();
    }

    /**
     * Requests Materials if required.
     * - If the entity is a builder.
     * - If the builder doesn't have infinite resources.
     */
    private void requestMaterialsIfRequired()
    {
        if (!Configurations.builderInfiniteResources && job instanceof JobBuilder && getOwnBuilding() instanceof BuildingBuilder)
        {
            ((BuildingBuilder) getOwnBuilding()).resetNeededResources();
            requestMaterials();
        }
    }

    /**
     * Searches a handy block to substitute a non-solid space which should be guaranteed solid.
     *
     * @param location the location the block should be at.
     * @return the Block.
     */
    public abstract IBlockState getSolidSubstitution(BlockPos location);

    /**
     * Loads the structure given the name, rotation and position.
     *
     * @param name        the name to retrieve  it.
     * @param rotateTimes number of times to rotateWithMirror it.
     * @param position    the position to set it.
     * @param isMirrored  is the structure mirroed?
     */
    public void loadStructure(@NotNull final String name, int rotateTimes, BlockPos position, boolean isMirrored)
    {
        if (job instanceof AbstractJobStructure)
        {
            rotation = rotateTimes;
            try
            {
                final StructureWrapper wrapper = new StructureWrapper(world, name);

                ((AbstractJobStructure) job).setStructure(wrapper);
                currentStructure = new Structure(world, wrapper, Structure.Stage.CLEAR);
            }
            catch (final IllegalStateException e)
            {
                Log.getLogger().warn(String.format("StructureProxy: (%s) does not exist - removing build request", name), e);
                ((AbstractJobStructure) job).setStructure(null);
            }

            ((AbstractJobStructure) job).getStructure().rotate(rotateTimes, world, position, isMirrored ? Mirror.FRONT_BACK : Mirror.NONE);
            ((AbstractJobStructure) job).getStructure().setPosition(position);
        }
    }

    /**
     * Check if the structure tusk has been canceled.
     * @return true if reset to idle.
     */
    protected abstract boolean checkIfCanceled();

    /**
     * Iterates through all the required resources and stores them in the building.
     */
    private void requestMaterials()
    {
        final JobBuilder builderJob = (JobBuilder) job;
        while (builderJob.getStructure().findNextBlock())
        {
            @Nullable final Template.BlockInfo blockInfo = builderJob.getStructure().getBlockInfo();
            @Nullable final Template.EntityInfo entityInfo = builderJob.getStructure().getEntityinfo();

            if (blockInfo == null)
            {
                continue;
            }

            requestEntityToBuildingIfRequired(entityInfo);

            @Nullable final IBlockState blockState = blockInfo.field_186243_b;
            @Nullable final Block block = blockState.func_177230_c();

            if (builderJob.getStructure().doesStructureBlockEqualWorldBlock()
                    || (blockState.func_177230_c() instanceof BlockBed && blockState.func_177229_b(BlockBed.field_176472_a).equals(BlockBed.EnumPartType.FOOT))
                    || (blockState.func_177230_c() instanceof BlockDoor && blockState.func_177229_b(BlockDoor.field_176523_O).equals(BlockDoor.EnumDoorHalf.UPPER)))
            {
                continue;
            }

            final Block worldBlock = BlockPosUtil.getBlock(world, builderJob.getStructure().getBlockPosition());

            if (block != null
                    && block != Blocks.field_150350_a
                    && worldBlock != Blocks.field_150357_h
                    && !(worldBlock instanceof AbstractBlockHut)
                    && !isBlockFree(block, 0))
            {
                final AbstractBuilding building = getOwnBuilding();
                if (building instanceof BuildingBuilder)
                {
                    requestBlockToBuildingIfRequired((BuildingBuilder) building, blockState);
                }
            }
        }
        builderJob.getWorkOrder().setRequested(true);
    }

    /**
     * Add blocks to the builder building if he needs it.
     *
     * @param building   the building.
     * @param blockState the block to add.
     */
    private void requestBlockToBuildingIfRequired(BuildingBuilder building, IBlockState blockState)
    {
        if (((JobBuilder) job).getStructure().getBlockInfo().field_186244_c != null)
        {
            final List<ItemStack> itemList = new ArrayList<>();
            itemList.addAll(getItemStacksOfTileEntity(((JobBuilder) job).getStructure().getBlockInfo().field_186244_c));

            for (final ItemStack stack : itemList)
            {
                building.addNeededResource(stack, 1);
            }
        }

        building.addNeededResource(BlockUtils.getItemStackFromBlockState(blockState), 1);
    }

    /**
     * Adds entities to the builder building if he needs it.
     */
    private void requestEntityToBuildingIfRequired(Template.EntityInfo entityInfo)
    {
        if (entityInfo != null)
        {
            final Entity entity = getEntityFromEntityInfoOrNull(entityInfo);

            if (entity != null)
            {
                final List<ItemStack> request = new ArrayList<>();
                if (entity instanceof EntityItemFrame)
                {
                    final ItemStack stack = ((EntityItemFrame) entity).func_82335_i();
                    if (stack != null)
                    {
                        stack.field_77994_a = 1;
                        request.add(stack);
                        request.add(new ItemStack(Items.field_151160_bD, 1, stack.func_77952_i()));
                    }
                }
                else if (entity instanceof EntityArmorStand)
                {
                    request.add(entity.getPickedResult(new RayTraceResult(worker)));
                    entity.func_184193_aE().forEach(request::add);
                }
                else
                {
                    request.add(entity.getPickedResult(new RayTraceResult(worker)));
                }

                for (final ItemStack stack : request)
                {
                    final AbstractBuilding building = getOwnBuilding();
                    if (building instanceof BuildingBuilder && stack != null && stack.func_77973_b() != null)
                    {
                        ((BuildingBuilder) building).addNeededResource(stack, 1);
                    }
                }
            }
        }
    }

    /**
     * Works on clearing the area of unneeded blocks.
     *
     * @return the next step once done.
     */
    private boolean clearStep(@NotNull final Structure.StructureBlock currentBlock)
    {
        if ((job instanceof JobBuilder && ((JobBuilder) job).getWorkOrder() != null && ((JobBuilder) job).getWorkOrder().isCleared())
                || !currentStructure.getStage().equals(Structure.Stage.CLEAR))
        {
            return true;
        }

        //Don't break bedrock etc.
        if (!BlockUtils.shouldNeverBeMessedWith(currentBlock.worldBlock))
        {
            //Fill workFrom with the position from where the builder should build.
            //also ensure we are at that position.
            if (!walkToConstructionSite())
            {
                return false;
            }

            worker.faceBlock(currentBlock.blockPosition);

            //We need to deal with materials
            if (Configurations.builderInfiniteResources || currentBlock.worldMetadata.func_185904_a().func_76224_d())
            {
                worker.func_184201_a(EntityEquipmentSlot.MAINHAND, null);
                world.func_175698_g(currentBlock.blockPosition);
                world.func_175656_a(currentBlock.blockPosition, Blocks.field_150350_a.func_176223_P());
                worker.func_184609_a(worker.func_184600_cs());
                setDelay(UNLIMITED_RESOURCES_TIMEOUT);
            }
            else
            {
                if (!mineBlock(currentBlock.blockPosition, workFrom == null ? getWorkingPosition() : workFrom))
                {
                    return false;
                }
            }
        }

        return true;
    }

    /**
     * Gets a floorPosition in a particular direction.
     *
     * @param facing   the direction.
     * @param distance the distance.
     * @return a BlockPos position.
     */
    @NotNull
    private BlockPos getPositionInDirection(final EnumFacing facing, final int distance)
    {
        return getFloor(currentStructure.getCurrentBlockPosition().func_177967_a(facing, distance));
    }

    /**
     * Calculates the floor level.
     *
     * @param position input position.
     * @return returns BlockPos position with air above.
     */
    @NotNull
    private BlockPos getFloor(@NotNull BlockPos position)
    {
        final BlockPos floor = getFloor(position, 0);
        if (floor == null)
        {
            return position;
        }
        return floor;
    }

    /**
     * Calculates the floor level.
     *
     * @param position input position.
     * @param depth    the iteration depth.
     * @return returns BlockPos position with air above.
     */
    @Nullable
    private BlockPos getFloor(@NotNull BlockPos position, int depth)
    {
        if (depth > 50)
        {
            return null;
        }
        //If the position is floating in Air go downwards
        if (!EntityUtils.solidOrLiquid(world, position))
        {
            return getFloor(position.func_177977_b(), depth + 1);
        }
        //If there is no air above the block go upwards
        if (!EntityUtils.solidOrLiquid(world, position.func_177984_a()))
        {
            return position;
        }
        return getFloor(position.func_177984_a(), depth + 1);
    }

    /**
     * Check if there is a Structure to be build.
     *
     * @return true if we should start building.
     */
    protected boolean isThereAStructureToBuild()
    {
        return currentStructure != null;
    }

    /**
     * Walk to the current construction site.
     * <p>
     * Calculates and caches the position where to walk to.
     *
     * @return true while walking to the site.
     */
    public boolean walkToConstructionSite()
    {
        if (workFrom == null)
        {
            workFrom = getWorkingPosition();
        }

        //The miner shouldn't search for a save position. Just let him build from where he currently is.
        return worker.isWorkerAtSiteWithMove(workFrom, STANDARD_WORKING_RANGE) || MathUtils.twoDimDistance(worker.func_180425_c(), workFrom) < MIN_WORKING_RANGE;
    }

    /**
     * Calculates the working position.
     * <p>
     * Takes a min distance from width and length.
     * <p>
     * Then finds the floor level at that distance and then check if it does contain two air levels.
     *
     * @return BlockPos position to work from.
     */
    private BlockPos getWorkingPosition()
    {
        if (job instanceof JobMiner)
        {
            return getNodeMiningPosition(currentStructure.getCurrentBlockPosition());
        }
        return getWorkingPosition(0);
    }

    /**
     * Calculates the working position.
     * <p>
     * Takes a min distance from width and length.
     * <p>
     * Then finds the floor level at that distance and then check if it does contain two air levels.
     *
     * @param offset the extra distance to apply away from the building.
     * @return BlockPos position to work from.
     */
    private BlockPos getWorkingPosition(final int offset)
    {
        if (offset > MAX_ADDITIONAL_RANGE_TO_BUILD)
        {
            return currentStructure.getCurrentBlockPosition();
        }
        //get length or width either is larger.
        final int length = currentStructure.getLength();
        final int width = currentStructure.getWidth();
        final int distance = Math.max(width, length) + MIN_ADDITIONAL_RANGE_TO_BUILD + offset;
        @NotNull final EnumFacing[] directions = {EnumFacing.EAST, EnumFacing.WEST, EnumFacing.NORTH, EnumFacing.SOUTH};

        //then get a solid place with two air spaces above it in any direction.
        for (final EnumFacing direction : directions)
        {
            @NotNull final BlockPos positionInDirection = getPositionInDirection(direction, distance);
            if (EntityUtils.checkForFreeSpace(world, positionInDirection))
            {
                return positionInDirection;
            }
        }

        //if necessary we can could implement calling getWorkingPosition recursively and add some "offset" to the sides.
        return getWorkingPosition(offset + 1);
    }

    /**
     * Start building this Structure.
     * <p>
     * Will determine where to start.
     *
     * @return the new State to start in.
     */
    @NotNull
    private AIState startBuilding()
    {
        if (currentStructure == null)
        {
            onStartWithoutStructure();
            return AIState.IDLE;
        }
        switch (currentStructure.getStage())
        {
            case CLEAR:
                return AIState.CLEAR_STEP;
            case BUILD:
                return AIState.BUILDING_STEP;
            case DECORATE:
                return AIState.DECORATION_STEP;
            case SPAWN:
                return AIState.SPAWN_STEP;
            default:
                return AIState.COMPLETE_BUILD;
        }
    }

    protected abstract void onStartWithoutStructure();

    private boolean handleMaterials(@NotNull final Block block, @NotNull final IBlockState blockState)
    {
        //Breaking blocks doesn't require taking materials from the citizens inventory
        if (block == Blocks.field_150350_a)
        {
            return true;
        }

        final List<ItemStack> itemList = new ArrayList<>();
        itemList.add(BlockUtils.getItemStackFromBlockState(blockState));
        if (job instanceof JobBuilder && ((JobBuilder) job).getStructure() != null
                && ((JobBuilder) job).getStructure().getBlockInfo() != null && ((JobBuilder) job).getStructure().getBlockInfo().field_186244_c != null)
        {
            itemList.addAll(getItemStacksOfTileEntity(((JobBuilder) job).getStructure().getBlockInfo().field_186244_c));
        }

        for (final ItemStack stack : itemList)
        {
            if (stack != null && checkOrRequestItems(getTotalAmount(stack)))
            {
                return false;
            }
        }

        return true;
    }

    /**
     * Check how much of a certain stuck is actually required.
     *
     * @param stack the stack to check.
     * @return the new stack with the correct amount.
     */
    @Nullable
    private ItemStack getTotalAmount(@Nullable final ItemStack stack)
    {
        final AbstractBuildingWorker buildingWorker = getOwnBuilding();
        if (buildingWorker instanceof BuildingBuilder)
        {
            final BuildingBuilderResource resource = ((BuildingBuilder) buildingWorker).getNeededResources().get(stack.func_77977_a());
            return resource == null ? stack : new ItemStack(resource.getItem(), resource.getAmount(), resource.getDamageValue());
        }
        return stack;
    }

    /**
     * Get itemStack of tileEntityData. Retrieve the data from the tileEntity.
     *
     * @param compound the tileEntity stored in a compound.
     * @return the list of itemstacks.
     */
    private List<ItemStack> getItemStacksOfTileEntity(NBTTagCompound compound)
    {
        final List<ItemStack> items = new ArrayList<>();
        final TileEntity tileEntity = TileEntity.func_190200_a(world, compound);
        if (tileEntity instanceof TileEntityFlowerPot)
        {
            items.add(((TileEntityFlowerPot) tileEntity).func_184403_b());
        }
        else if (tileEntity instanceof TileEntityLockable)
        {
            for (int i = 0; i < ((TileEntityLockable) tileEntity).func_70302_i_(); i++)
            {
                final ItemStack stack = ((TileEntityLockable) tileEntity).func_70301_a(i);
                if (stack != null)
                {
                    items.add(stack);
                }
            }
        }
        return items;
    }

    /**
     * Defines blocks that can be built for free.
     *
     * @param block    The block to check if it is free.
     * @param metadata The metadata of the block.
     * @return true or false.
     */
    public static boolean isBlockFree(@Nullable final Block block, final int metadata)
    {
        return block == null
                || BlockUtils.isWater(block.func_176223_P())
                || block.equals(Blocks.field_150362_t)
                || block.equals(Blocks.field_150361_u)
                || (block.equals(Blocks.field_150398_cm) && Utils.testFlag(metadata, 0x08))
                || block.equals(Blocks.field_150349_c)
                || block.equals(ModBlocks.blockSolidSubstitution);
    }

    private void placeBlockAt(@NotNull final Block block, @NotNull final IBlockState blockState, @NotNull final BlockPos coords)
    {
        if (block == Blocks.field_150350_a)
        {
            worker.func_184201_a(EntityEquipmentSlot.MAINHAND, null);

            if (!world.func_175698_g(coords))
            {
                Log.getLogger().error(String.format("Block break failure at %s", coords));
            }
        }
        else
        {
            final ItemStack item = BlockUtils.getItemStackFromBlockState(blockState);
            worker.func_184201_a(EntityEquipmentSlot.MAINHAND, item == null ? null : item);

            if (!placeBlock(coords, block, blockState))
            {
                Log.getLogger().error(String.format("Block place failure %s at %s", block.func_149739_a(), coords));
            }
            worker.func_184609_a(worker.func_184600_cs());
        }
    }

    private boolean placeBlock(@NotNull final BlockPos pos, final Block block, @NotNull final IBlockState blockState)
    {
        //Move out of the way when placing blocks
        if (MathHelper.func_76128_c(worker.field_70165_t) == pos.func_177958_n()
                && MathHelper.func_76130_a(pos.func_177956_o() - (int) worker.field_70163_u) <= 1
                && MathHelper.func_76128_c(worker.field_70161_v) == pos.func_177952_p()
                && worker.func_70661_as().func_75500_f())
        {
            worker.func_70661_as().moveAwayFromXYZ(pos, RUN_AWAY_SPEED, 1.0);
        }

        //Workaround as long as we didn't rescan all of our buildings since BlockStairs now have different metadata values.
        if (blockState.func_177230_c() instanceof BlockStairs
                && world.func_180495_p(pos).func_177230_c() instanceof BlockStairs
                && world.func_180495_p(pos).func_177229_b(BlockStairs.field_176309_a) == blockState.func_177229_b(BlockStairs.field_176309_a)
                && blockState.func_177230_c() == world.func_180495_p(pos).func_177230_c())
        {
            return true;
        }

        //We need to deal with materials
        if (!Configurations.builderInfiniteResources && world.func_180495_p(pos).func_177230_c() != Blocks.field_150350_a)
        {
            final List<ItemStack> items = BlockPosUtil.getBlockDrops(world, pos, 0);
            for (final ItemStack item : items)
            {
                InventoryUtils.setStack(worker.getInventoryCitizen(), item);
            }
        }

        if (block instanceof BlockDoor)
        {
            if (blockState.func_177229_b(BlockDoor.field_176523_O).equals(BlockDoor.EnumDoorHalf.LOWER))
            {
                ItemDoor.func_179235_a(world, pos, blockState.func_177229_b(BlockDoor.field_176520_a), block, false);
            }
        }
        else if (block instanceof BlockBed)
        {
            final EnumFacing facing = blockState.func_177229_b(BlockBed.field_185512_D);

            //Set other part of the bed, to the opposite PartType
            if (blockState.func_177229_b(BlockBed.field_176472_a) == BlockBed.EnumPartType.FOOT)
            {
                //pos.offset(facing) will get the other part of the bed
                world.func_180501_a(pos.func_177972_a(facing), blockState.func_177226_a(BlockBed.field_176472_a, BlockBed.EnumPartType.HEAD), 0x03);
                world.func_180501_a(pos, blockState.func_177226_a(BlockBed.field_176472_a, BlockBed.EnumPartType.FOOT), 0x03);
            }
            else
            {
                return true;
            }
        }
        else if (block instanceof BlockDoublePlant)
        {
            world.func_180501_a(pos, blockState.func_177226_a(BlockDoublePlant.field_176492_b, BlockDoublePlant.EnumBlockHalf.LOWER), 0x03);
            world.func_180501_a(pos.func_177984_a(), blockState.func_177226_a(BlockDoublePlant.field_176492_b, BlockDoublePlant.EnumBlockHalf.UPPER), 0x03);
        }
        else if (block instanceof BlockEndPortal || block instanceof BlockMobSpawner || block instanceof BlockDragonEgg || block instanceof BlockPortal)
        {
            return true;
        }
        else if (block instanceof BlockFlowerPot)
        {
            if (!world.func_180501_a(pos, blockState, 0x03))
            {
                return false;
            }

            //This creates the flowerPot tileEntity from its BlockInfo to set the required data into the world.
            if (job instanceof JobBuilder && ((JobBuilder) job).getStructure().getBlockInfo().field_186244_c != null)
            {
                final TileEntityFlowerPot tileentityflowerpot = (TileEntityFlowerPot) world.func_175625_s(pos);
                tileentityflowerpot.func_145839_a(((JobBuilder) job).getStructure().getBlockInfo().field_186244_c);
                world.func_175690_a(pos, tileentityflowerpot);
            }
        }
        else
        {
            if (!world.func_180501_a(pos, blockState, 0x03))
            {
                return false;
            }
        }

        if(block instanceof BlockChest && job instanceof JobBuilder)
        {
            final BlockPos buildingLocation = ((JobBuilder) job).getWorkOrder().getBuildingLocation();
            final AbstractBuilding building = this.getOwnBuilding().getColony().getBuilding(buildingLocation);
            building.addContainerPosition(pos);
        }

        //It will crash at blocks like water which is actually free, we don't have to decrease the stacks we have.
        if (isBlockFree(block, block.func_176201_c(blockState)))
        {
            return true;
        }

        @Nullable final ItemStack stack = BlockUtils.getItemStackFromBlockState(blockState);
        if (stack == null)
        {
            Log.getLogger().error("Block causes NPE: " + blockState.func_177230_c());
            return false;
        }

        final List<ItemStack> itemList = new ArrayList<>();
        itemList.add(stack);
        if (job instanceof JobBuilder && ((JobBuilder) job).getStructure() != null
                && ((JobBuilder) job).getStructure().getBlockInfo() != null && ((JobBuilder) job).getStructure().getBlockInfo().field_186244_c != null)
        {
            itemList.addAll(getItemStacksOfTileEntity(((JobBuilder) job).getStructure().getBlockInfo().field_186244_c));
        }

        for (final ItemStack tempStack : itemList)
        {
            if (tempStack != null)
            {
                final int slot = worker.findFirstSlotInInventoryWith(tempStack.func_77973_b(), tempStack.func_77952_i());
                if (slot != -1)
                {
                    getInventory().func_70298_a(slot, 1);
                    reduceNeededResources(tempStack);
                }
            }
        }

        if (Configurations.builderBuildBlockDelay > 0 && block != Blocks.field_150350_a)
        {
            setDelay(Configurations.builderBuildBlockDelay);
        }

        return true;
    }

    /**
     * Reduces the needed resources by 1.
     *
     * @param stack the stack which has been used now.
     */
    public void reduceNeededResources(final ItemStack stack)
    {
        final AbstractBuilding workerBuilding = this.getOwnBuilding();
        if (workerBuilding instanceof BuildingBuilder)
        {
            ((BuildingBuilder) workerBuilding).reduceNeededResource(stack, 1);
        }
    }

    /**
     * Get the entity of an entityInfo object.
     *
     * @param entityInfo the input.
     * @return the output object or null.
     */
    @Nullable
    private Entity getEntityFromEntityInfoOrNull(Template.EntityInfo entityInfo)
    {
        try
        {
            return EntityList.func_75615_a(entityInfo.field_186249_c, world);
        }
        catch (RuntimeException e)
        {
            Log.getLogger().info("Couldn't restore entitiy", e);
            return null;
        }
    }

    /**
     * Set the currentStructure to null.
     */
    public void resetCurrentStructure()
    {
        currentStructure = null;
    }

    private Boolean spawnEntity(@NotNull final Structure.StructureBlock currentBlock)
    {
        final Template.EntityInfo entityInfo = currentBlock.entity;
        if (entityInfo != null && job instanceof JobBuilder && ((JobBuilder) job).getStructure() != null)
        {
            final Entity entity = getEntityFromEntityInfoOrNull(entityInfo);

            if (entity != null && !isEntityAtPosition(entity, world))
            {
                final List<ItemStack> request = new ArrayList<>();

                if (entity instanceof EntityItemFrame)
                {
                    final ItemStack stack = ((EntityItemFrame) entity).func_82335_i();
                    if (stack != null)
                    {
                        stack.field_77994_a++;
                        request.add(stack);
                    }
                    request.add(new ItemStack(Items.field_151160_bD, 1));
                }
                else if (entity instanceof EntityArmorStand)
                {
                    request.add(entity.getPickedResult(new RayTraceResult(worker)));
                    entity.func_184193_aE().forEach(request::add);
                }
                else
                {
                    request.add(entity.getPickedResult(new RayTraceResult(worker)));
                }

                if (!Configurations.builderInfiniteResources)
                {
                    for (final ItemStack stack : request)
                    {
                        if (checkOrRequestItems(stack))
                        {
                            return false;
                        }
                    }
                    for (final ItemStack stack : request)
                    {
                        if (stack == null)
                        {
                            continue;
                        }
                        final int slot = worker.findFirstSlotInInventoryWith(stack.func_77973_b(), stack.func_77952_i());
                        if (slot != -1)
                        {
                            getInventory().func_70298_a(slot, 1);
                            reduceNeededResources(stack);
                        }
                    }
                }

                entity.func_184221_a(UUID.randomUUID());
                entity.func_70012_b(
                        entity.field_70165_t,
                        entity.field_70163_u,
                        entity.field_70161_v,
                        entity.field_70177_z,
                        entity.field_70125_A);
                if (!world.func_72838_d(entity))
                {
                    Log.getLogger().info("Failed to spawn entity");
                }
            }
        }
        return true;
    }

    /**
     * Checks if a certain entity is in the world at a certain position already.
     * @param entity the entity.
     * @param world the world.
     * @return true if there.
     */
    private static boolean isEntityAtPosition(final Entity entity, final World world)
    {
        if(world.func_72872_a(entity.getClass(), new AxisAlignedBB(entity.field_70165_t, entity.field_70163_u, entity.field_70161_v, entity.field_70165_t, entity.field_70163_u, entity.field_70161_v)).isEmpty())
        {
            return false;
        }

        return true;
    }

    /**
     * Create a save mining position for the miner.
     *
     * @param blockToMine block which should be mined or placed.
     * @return the save position.
     */
    private BlockPos getNodeMiningPosition(BlockPos blockToMine)
    {
        if (getOwnBuilding() instanceof BuildingMiner)
        {
            BuildingMiner buildingMiner = (BuildingMiner) getOwnBuilding();
            if (buildingMiner.getCurrentLevel() == null || buildingMiner.getCurrentLevel().getRandomNode() == null)
            {
                return blockToMine;
            }
            final Point2D parentPos = buildingMiner.getCurrentLevel().getRandomNode().getParent();
            if(parentPos != null && buildingMiner.getCurrentLevel().getNode(parentPos) != null
                    && buildingMiner.getCurrentLevel().getNode(parentPos).getStyle() == Node.NodeType.SHAFT)
            {
                final BlockPos ladderPos = buildingMiner.getLadderLocation();
                return new BlockPos(
                        ladderPos.func_177958_n() + buildingMiner.getVectorX() * OTHER_SIDE_OF_SHAFT,
                        buildingMiner.getCurrentLevel().getDepth(),
                        ladderPos.func_177952_p() + buildingMiner.getVectorZ() * OTHER_SIDE_OF_SHAFT);
            }
            final Point2D pos = buildingMiner.getCurrentLevel().getRandomNode().getParent();
            return new BlockPos(pos.getX(), buildingMiner.getCurrentLevel().getDepth(), pos.getY());
        }
        return blockToMine;
    }
}
