package com.minecolonies.coremod.entity.ai.citizen.builder;

import com.minecolonies.coremod.blocks.AbstractBlockHut;
import com.minecolonies.coremod.blocks.ModBlocks;
import com.minecolonies.coremod.colony.buildings.AbstractBuilding;
import com.minecolonies.coremod.colony.workorders.WorkOrderBuild;
import com.minecolonies.coremod.colony.workorders.WorkOrderBuildDecoration;
import com.minecolonies.coremod.configuration.Configurations;
import com.minecolonies.coremod.util.BlockUtils;
import com.minecolonies.coremod.util.StructureWrapper;
import net.minecraft.block.Block;
import net.minecraft.block.BlockHorizontal;
import net.minecraft.block.properties.PropertyDirection;
import net.minecraft.block.state.IBlockState;
import net.minecraft.init.Blocks;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.Mirror;
import net.minecraft.util.math.BlockPos;
import net.minecraft.world.World;
import org.jetbrains.annotations.NotNull;

/**
 * Helper class to place and remove constructionTapes from the buildings.
 */
public final class ConstructionTapeHelper
{
    public static final  PropertyDirection FACING     = BlockHorizontal.field_185512_D;
    public static final int MINHEIGHT = 1;
    public static final int MAXHEIGHT = 256;
    /**
     * Private Constructor to hide implicit one.
     */
    private ConstructionTapeHelper()
    {
        /**
         * Intentionally left empty.
         */
    }

    /**
     * Check if a block is placeable and return new Y position.
     * @param x Block X position.
     * @param y Block Y position.
     * @param z Block Z position.
     * @param world the world.
     * @return The new Y position.
     */

    public static int checkIfPlaceable(@NotNull int x, @NotNull int y, @NotNull int z, @NotNull World world)
    {
        int newY = y;
        boolean working = true;
        while (working)
        {
            final BlockPos block = new BlockPos(x, newY, z);
            final BlockPos blockMin1 = new BlockPos(x, newY-1, z);
            if (world.func_180495_p(block).func_185904_a().func_76222_j())
            {
                if (world.func_180495_p(blockMin1).func_185904_a().func_76222_j() && newY >= 1)
                {
                    newY = newY-1;
                }
                else
                {
                    working = false;
                }
            }
            else
            {
                newY = newY+1;
            }
        }
        return newY > 0 ? newY : y;
    }

    /**
     * @param world the world.
     * @param block the block.
     * @param tapeOrTapeCorner Is the checked block supposed to be ConstructionTape or ConstructionTapeCorner.
     */
    public static void removeTapeIfNecessary(@NotNull World world,@NotNull BlockPos block,@NotNull Block tapeOrTapeCorner)
    {
        for (int y = MINHEIGHT; y <= MAXHEIGHT; y++)
        {
            final BlockPos newBlock = new BlockPos(block.func_177958_n(), y, block.func_177952_p());
            if (world.func_180495_p(newBlock).func_177230_c() == tapeOrTapeCorner)
            {
                world.func_175656_a(newBlock, Blocks.field_150350_a.func_176223_P());
                break;
            }
        }
    }

    /**
     * Proxy to place the tape also with the building only.
     * @param building the building.
     * @param world the world.
     */
    public static void placeConstructionTape(@NotNull AbstractBuilding building, @NotNull World world)
    {
        placeConstructionTape(new WorkOrderBuild(building, 1), world);
    }

    /**
     * Place construction tape.
     * @param workOrder the workorder.
     * @param world the world.
     */

    public static void placeConstructionTape(@NotNull WorkOrderBuild workOrder, @NotNull World world)
    {
        if (Configurations.builderPlaceConstructionTape)
        {
            final StructureWrapper wrapper = new StructureWrapper(world, workOrder.getStructureName());
            final BlockPos pos = workOrder.getBuildingLocation();
            int tempRotation = 0;
            final IBlockState constructionTape = ModBlocks.blockConstructionTape.func_176223_P();
            final IBlockState constructionTapeCorner = ModBlocks.blockConstructionTapeCorner.func_176223_P();

            if (workOrder.getRotation() == 0 && !(workOrder instanceof WorkOrderBuildDecoration))
            {
                final IBlockState blockState = world.func_180495_p(pos);
                if (blockState.func_177230_c() instanceof AbstractBlockHut)
                {
                    tempRotation = BlockUtils.getRotationFromFacing(blockState.func_177229_b(AbstractBlockHut.FACING));
                }
            }
            else
            {
                tempRotation = workOrder.getRotation();
            }

            wrapper.rotate(tempRotation, world, workOrder.getBuildingLocation(), workOrder.isMirrored() ? Mirror.FRONT_BACK : Mirror.NONE);
            wrapper.setPosition(pos);

            final int x1 = wrapper.getPosition().func_177958_n() - wrapper.getOffset().func_177958_n() - 1;
            final int z1 = wrapper.getPosition().func_177952_p() - wrapper.getOffset().func_177952_p() - 1;
            final int x3 = wrapper.getPosition().func_177958_n() + (wrapper.getWidth() - wrapper.getOffset().func_177958_n());
            final int z3 = wrapper.getPosition().func_177952_p() + (wrapper.getLength() - wrapper.getOffset().func_177952_p());
            final int y = wrapper.getPosition().func_177956_o();
            int newY;

            if (x1 < x3)
            {
                for (int i = x1 + 1; i < x3; i++)
                {
                    newY = checkIfPlaceable(i, y, z1, world);
                    final BlockPos row1 = new BlockPos(i, newY, z1);
                    world.func_175656_a(row1, constructionTape.func_177226_a(FACING, EnumFacing.SOUTH));
                    newY = checkIfPlaceable(i, y, z3, world);
                    final BlockPos row2 = new BlockPos(i, newY, z3);
                    world.func_175656_a(row2, constructionTape.func_177226_a(FACING, EnumFacing.NORTH));
                }
            }
            else
            {
                for (int i = x3 + 1; i < x1; i++)
                {
                    newY = checkIfPlaceable(i, y, z1, world);
                    final BlockPos row1 = new BlockPos(i, newY, z1);
                    world.func_175656_a(row1, constructionTape.func_177226_a(FACING, EnumFacing.SOUTH));
                    newY = checkIfPlaceable(i, y, z3, world);
                    final BlockPos row2 = new BlockPos(i, newY, z3);
                    world.func_175656_a(row2, constructionTape.func_177226_a(FACING, EnumFacing.NORTH));
                }
            }
            if (z1 < z3)
            {
                for (int i = z1 + 1; i < z3; i++)
                {
                    newY = checkIfPlaceable(x1, y, i, world);
                    final BlockPos row3 = new BlockPos(x1, newY, i);
                    world.func_175656_a(row3, constructionTape.func_177226_a(FACING, EnumFacing.EAST));
                    newY = checkIfPlaceable(x3, y, i, world);
                    final BlockPos row4 = new BlockPos(x3, newY, i);
                    world.func_175656_a(row4, constructionTape.func_177226_a(FACING, EnumFacing.WEST));
                }
            }
            else
            {
                for (int i = z3 + 1; i < z1; i++)
                {
                    newY = checkIfPlaceable(x1, y, i, world);
                    final BlockPos row3 = new BlockPos(x1, newY, i);
                    world.func_175656_a(row3, constructionTape.func_177226_a(FACING, EnumFacing.EAST));
                    newY = checkIfPlaceable(x3, y, i, world);
                    final BlockPos row4 = new BlockPos(x3, newY, i);
                    world.func_175656_a(row4, constructionTape.func_177226_a(FACING, EnumFacing.WEST));
                }
            }
            newY = checkIfPlaceable(x1, y, z1, world);
            final BlockPos corner1 = new BlockPos(x1, newY, z1);
            newY = checkIfPlaceable(x1, y, z3, world);
            final BlockPos corner2 = new BlockPos(x1, newY, z3);
            newY = checkIfPlaceable(x3, y, z1, world);
            final BlockPos corner3 = new BlockPos(x3, newY, z1);
            newY = checkIfPlaceable(x3, y, z3, world);
            final BlockPos corner4 = new BlockPos(x3, newY, z3);
            world.func_175656_a(corner1, constructionTapeCorner.func_177226_a(FACING, EnumFacing.SOUTH));
            world.func_175656_a(corner2, constructionTapeCorner.func_177226_a(FACING, EnumFacing.EAST));
            world.func_175656_a(corner3, constructionTapeCorner.func_177226_a(FACING, EnumFacing.WEST));
            world.func_175656_a(corner4, constructionTapeCorner.func_177226_a(FACING, EnumFacing.NORTH));
        }
    }

    /**
     * Proxy to remove the tape also with the building only.
     * @param building the building.
     * @param world the world.
     */
    public static void removeConstructionTape(@NotNull AbstractBuilding building, @NotNull World world)
    {
        removeConstructionTape(new WorkOrderBuild(building, 1), world);
    }

    /**
     * Remove construction tape.
     * @param workOrder the workorder.
     * @param world the world.
     */

    public static void removeConstructionTape(@NotNull WorkOrderBuild workOrder,@NotNull World world)
    {
        final StructureWrapper wrapper = new StructureWrapper(world, workOrder.getStructureName());
        final BlockPos pos = workOrder.getBuildingLocation();
        int tempRotation = 0;
        if (workOrder.getRotation() == 0 && !(workOrder instanceof WorkOrderBuildDecoration))
        {
            final IBlockState blockState = world.func_180495_p(pos);
            if (blockState.func_177230_c() instanceof AbstractBlockHut)
            {
                tempRotation = BlockUtils.getRotationFromFacing(blockState.func_177229_b(AbstractBlockHut.FACING));
            }
        }
        else
        {
            tempRotation = workOrder.getRotation();
        }
        wrapper.rotate(tempRotation, world, workOrder.getBuildingLocation(), workOrder.isMirrored() ? Mirror.FRONT_BACK : Mirror.NONE);
        wrapper.setPosition(pos);
        final int x1 = wrapper.getPosition().func_177958_n() - wrapper.getOffset().func_177958_n() - 1;
        final int z1 = wrapper.getPosition().func_177952_p() - wrapper.getOffset().func_177952_p() - 1;
        final int x3 = wrapper.getPosition().func_177958_n() + (wrapper.getWidth() - wrapper.getOffset().func_177958_n());
        final int z3 = wrapper.getPosition().func_177952_p() + (wrapper.getLength() - wrapper.getOffset().func_177952_p());
        if (x1 < x3)
        {
            for (int i = x1; i <= x3; i++)
            {
                final BlockPos block1 = new BlockPos(i, 0, z1);
                final BlockPos block2 = new BlockPos(i, 0, z3);
                removeTapeIfNecessary(world,block1,ModBlocks.blockConstructionTape);
                removeTapeIfNecessary(world,block2,ModBlocks.blockConstructionTape);
            }
        }
        else
        {
            for (int i = x3; i <= x1; i++)
            {
                final BlockPos block1 = new BlockPos(i, 0, z1);
                final BlockPos block2 = new BlockPos(i, 0, z3);
                removeTapeIfNecessary(world,block1,ModBlocks.blockConstructionTape);
                removeTapeIfNecessary(world,block2,ModBlocks.blockConstructionTape);
            }
        }
        if (z1 < z3)
        {
            for (int i = z1; i <= z3; i++)
            {
                final BlockPos block1 = new BlockPos(x1, 0, i);
                final BlockPos block2 = new BlockPos(x3, 0, i);
                removeTapeIfNecessary(world,block1,ModBlocks.blockConstructionTape);
                removeTapeIfNecessary(world,block2,ModBlocks.blockConstructionTape);
            }
        }
        else
        {
            for (int i = z3; i <= z1; i++)
            {
                final BlockPos block1 = new BlockPos(x1, 0, i);
                final BlockPos block2 = new BlockPos(x3, 0, i);
                removeTapeIfNecessary(world,block1,ModBlocks.blockConstructionTape);
                removeTapeIfNecessary(world,block2,ModBlocks.blockConstructionTape);
            }
        }
            final BlockPos corner1 = new BlockPos(x1, 0, z1);
            final BlockPos corner2 = new BlockPos(x1, 0, z3);
            final BlockPos corner3 = new BlockPos(x3, 0, z1);
            final BlockPos corner4 = new BlockPos(x3, 0, z3);
            removeTapeIfNecessary(world,corner1,ModBlocks.blockConstructionTapeCorner);
            removeTapeIfNecessary(world,corner2,ModBlocks.blockConstructionTapeCorner);
            removeTapeIfNecessary(world,corner3,ModBlocks.blockConstructionTapeCorner);
            removeTapeIfNecessary(world,corner4,ModBlocks.blockConstructionTapeCorner);
    }
}
