package com.minecolonies.coremod.entity.ai.minimal;

import com.minecolonies.coremod.colony.jobs.JobGuard;
import com.minecolonies.coremod.entity.EntityCitizen;
import net.minecraft.entity.Entity;
import net.minecraft.entity.ai.EntityAIBase;
import net.minecraft.entity.player.EntityPlayer;
import org.jetbrains.annotations.Nullable;

import java.util.Optional;

/**
 * AI task to avoid an Entity class.
 */
public class EntityAICitizenAvoidEntity extends EntityAIBase
{
    /**
     * Defines how close the entity has to be to the mob to run away.
     */
    private static final double TOO_CLOSE_TO_MOB = 49D;
    /**
     * The entity we are attached to.
     */
    private final EntityCitizen           theEntity;
    private final double                  farSpeed;
    private final double                  nearSpeed;
    @Nullable
    private       Entity                  closestLivingEntity;
    private final float                   distanceFromEntity;
    private final Class<? extends Entity> targetEntityClass;

    /**
     * Constructor.
     *
     * @param entity             current entity.
     * @param targetEntityClass  entity class we want to avoid.
     * @param distanceFromEntity how far we want to stay away.
     * @param farSpeed           how fast we should move when we are far away.
     * @param nearSpeed          how fast we should move when we are close.
     */
    public EntityAICitizenAvoidEntity(
                                       final EntityCitizen entity, final Class<? extends Entity> targetEntityClass,
                                       final float distanceFromEntity, final double farSpeed, final double nearSpeed)
    {
        super();
        this.theEntity = entity;
        this.targetEntityClass = targetEntityClass;
        this.distanceFromEntity = distanceFromEntity;
        this.farSpeed = farSpeed;
        this.nearSpeed = nearSpeed;
        super.func_75248_a(1);
    }

    /**
     * Returns whether the EntityAIBase should begin execution of avoiding.
     */
    @Override
    public boolean func_75250_a()
    {
        closestLivingEntity = getClosestToAvoid();
        return closestLivingEntity != null && !(theEntity.getColonyJob() instanceof JobGuard) && this.theEntity.func_70685_l(closestLivingEntity);
    }

    /**
     * Returns the closest entity to avoid.
     *
     * @return Entity to avoid.
     */
    private Entity getClosestToAvoid()
    {
        if (targetEntityClass == EntityPlayer.class)
        {
            return theEntity.field_70170_p.func_72890_a(theEntity, (double) distanceFromEntity);
        }
        else
        {
            final Optional<Entity> entityOptional = theEntity.field_70170_p.func_175674_a(
              theEntity,
              theEntity.func_174813_aQ().func_72314_b(
                (double) distanceFromEntity,
                3.0D,
                (double) distanceFromEntity),
              target -> target.func_70089_S() && EntityAICitizenAvoidEntity.this.theEntity.func_70635_at().func_75522_a(target))
                                                      .stream()
                                                      .filter(targetEntityClass::isInstance)
                                                      .findFirst();

            return entityOptional.isPresent() ? entityOptional.get() : null;
        }
    }

    /**
     * Returns whether an in-progress EntityAIBase should continue executing.
     */
    @Override
    public boolean func_75253_b()
    {
        return !theEntity.func_70661_as().func_75500_f();
    }

    /**
     * Execute a one shot task or start executing a continuous task.
     */
    @Override
    public void func_75249_e()
    {
        performMoveAway();
    }

    /**
     * Makes entity move away from {@link #closestLivingEntity}.
     */
    private void performMoveAway()
    {
        theEntity.func_70661_as().moveAwayFromEntityLiving(closestLivingEntity, distanceFromEntity * 2D, nearSpeed);
    }

    /**
     * Resets the task.
     */
    @Override
    public void func_75251_c()
    {
        closestLivingEntity = null;
    }

    /**
     * Updates the task.
     */
    @Override
    public void func_75246_d()
    {
        @Nullable final Entity newClosest = getClosestToAvoid();
        if (newClosest != null && newClosest != closestLivingEntity)
        {
            closestLivingEntity = newClosest;
            performMoveAway();
            return;
        }

        if (theEntity.func_70068_e(closestLivingEntity) < TOO_CLOSE_TO_MOB)
        {
            theEntity.func_70661_as().func_75489_a(nearSpeed);
        }
        else
        {
            theEntity.func_70661_as().func_75489_a(farSpeed);
        }
    }
}
