package com.minecolonies.coremod.entity.pathfinding;

import com.minecolonies.coremod.entity.ai.citizen.lumberjack.Tree;
import net.minecraft.block.material.Material;
import net.minecraft.block.state.IBlockState;
import net.minecraft.util.math.BlockPos;
import net.minecraft.world.World;
import org.jetbrains.annotations.NotNull;

/**
 * Find and return a path to the nearest tree.
 * Created: May 21, 2015
 */
public class PathJobFindTree extends AbstractPathJob
{
    private final BlockPos hutLocation;

    /**
     * AbstractPathJob constructor.
     *
     * @param world the world within which to path.
     * @param start the start position from which to path from.
     * @param home  the position of the workers hut.
     * @param range maximum path range.
     */
    public PathJobFindTree(final World world, @NotNull final BlockPos start, final BlockPos home, final int range)
    {
        super(world, start, start, range, new TreePathResult());

        hutLocation = home;
    }

    /**
     * Custom result of the class which contains the position of the tree.
     */
    public static class TreePathResult extends PathResult
    {
        /**
         * Position of the found tree.
         */
        public BlockPos treeLocation;
    }

    @NotNull
    @Override
    public TreePathResult getResult()
    {
        return (TreePathResult) super.getResult();
    }

    @Override
    protected double computeHeuristic(@NotNull final BlockPos pos)
    {
        final int dx = pos.func_177958_n() - hutLocation.func_177958_n();
        final int dy = pos.func_177956_o() - hutLocation.func_177956_o();
        final int dz = pos.func_177952_p() - hutLocation.func_177952_p();

        //  Manhattan Distance with a 1/1000th tie-breaker - halved
        return (Math.abs(dx) + Math.abs(dy) + Math.abs(dz)) * 0.951D;
    }

    @Override
    protected boolean isAtDestination(@NotNull final Node n)
    {
        return n.parent != null && isNearTree(n);
    }

    private boolean isNearTree(@NotNull final Node n)
    {
        if (n.pos.func_177958_n() == n.parent.pos.func_177958_n())
        {
            final int dz = n.pos.func_177952_p() > n.parent.pos.func_177952_p() ? 1 : -1;
            return isTree(n.pos.func_177982_a(0, 0, dz)) || isTree(n.pos.func_177982_a(-1, 0, 0)) || isTree(n.pos.func_177982_a(1, 0, 0));
        }
        else
        {
            final int dx = n.pos.func_177958_n() > n.parent.pos.func_177958_n() ? 1 : -1;
            return isTree(n.pos.func_177982_a(-dx, 0, 0)) || isTree(n.pos.func_177982_a(0, 0, -1)) || isTree(n.pos.func_177982_a(0, 0, +1));
        }
    }

    private boolean isTree(final BlockPos pos)
    {
        if (Tree.checkTree(world, pos))
        {
            getResult().treeLocation = pos;
            return true;
        }

        return false;
    }

    @Override
    protected double getNodeResultScore(final Node n)
    {
        return 0;
    }

    @Override
    protected boolean isPassable(@NotNull final IBlockState block)
    {
        return super.isPassable(block) || block.func_185904_a() == Material.field_151584_j;
    }
}
