package com.minecolonies.coremod.entity.pathfinding;

import com.minecolonies.coremod.entity.EntityCitizen;
import com.minecolonies.coremod.util.BlockPosUtil;
import com.minecolonies.coremod.util.BlockUtils;
import com.minecolonies.coremod.util.Log;
import net.minecraft.entity.Entity;
import net.minecraft.entity.EntityLiving;
import net.minecraft.pathfinding.*;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.math.BlockPos;
import net.minecraft.util.math.MathHelper;
import net.minecraft.util.math.Vec3d;
import net.minecraft.world.World;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;

/**
 * Minecolonies async PathNavigate.
 */
public class PathNavigate extends PathNavigateGround
{
    public static final double MAX_PATHING_LENGTH          = 36.0;
    public static final double PATHING_INTERMEDIARY_LENGTH = 16.0;
    //  Parent class private members
    private final EntityLiving entity;
    private       double       walkSpeed;
    @Nullable
    private       BlockPos     destination;
    @Nullable
    private       BlockPos     originalDestination;
    @Nullable
    private       Future<Path> future;
    @Nullable
    private       PathResult   pathResult;

    /**
     * Instantiates the navigation of an entity.
     *
     * @param entity the entity.
     * @param world  the world it is in.
     */
    public PathNavigate(@NotNull final EntityLiving entity, final World world)
    {
        super(entity, world);
        this.entity = entity;

        this.field_179695_a = new WalkNodeProcessor();
    }

    @Nullable
    @Override
    protected PathFinder func_179679_a()
    {
        return null;
    }

    @Override
    protected boolean func_75485_k()
    {
        return true;
    }

    @Override
    protected Vec3d func_75502_i()
    {
        return this.entity.func_174791_d();
    }

    /**
     * Get the destination from the path.
     * @return the destionation position.
     */
    public BlockPos getDestination()
    {
        return destination;
    }

    @Nullable
    @Override
    public Path func_179680_a(final BlockPos pos)
    {
        //Because this directly returns Path we can't do it async.
        return null;
    }

    //  Re-enable this if path shortcutting becomes a problem; then entities will move more rigidly along world grid
//    @Override
//    protected boolean isDirectPathBetweenPoints(final Vec3d Vec3d, final Vec3d Vec3d1, final int i, final int i1, final int i2)
//    {
//        //we don't use, so it doesn't matter
//        return false;
//    }

    public double getSpeed()
    {
        return walkSpeed;
    }

    @Override
    public void func_75489_a(final double d)
    {
        walkSpeed = d;
    }

    @Override
    public boolean func_75492_a(final double x, final double y, final double z, final double speed)
    {
        moveToXYZ(x, y, z, speed);
        return true;
    }

    @Override
    public boolean func_75497_a(@NotNull final Entity e, final double speed)
    {
        return func_75492_a(e.field_70165_t, e.field_70163_u, e.field_70161_v, speed);
    }

    /**
     * Try to move to a certain position.
     *
     * @param x     the x target.
     * @param y     the y target.
     * @param z     the z target.
     * @param speed the speed to walk.
     * @return the PathResult.
     */
    @Nullable
    public PathResult moveToXYZ(final double x, final double y, final double z, final double speed)
    {
        int newX = MathHelper.func_76128_c(x);
        int newY = (int) y;
        int newZ = MathHelper.func_76128_c(z);


        if ((destination != null
               && BlockPosUtil.isEqual(destination, newX, newY, newZ))
              || (originalDestination != null
                    && BlockPosUtil.isEqual(originalDestination, newX, newY, newZ)
                    && pathResult != null
                    && pathResult.isInProgress()))
        {
            return pathResult;
        }

        final Vec3d moveVector = func_75502_i().func_72444_a(new Vec3d(newX, newY, newZ));
        final double moveLength = moveVector.func_72433_c();
        if (moveLength >= MAX_PATHING_LENGTH && !this.isUnableToReachDestination())
        {
            final Vec3d newMove = moveVector.func_186678_a(PATHING_INTERMEDIARY_LENGTH / moveLength).func_178787_e(func_75502_i());
            originalDestination = new BlockPos(newX, newY, newZ);
            newX = MathHelper.func_76128_c(newMove.field_72450_a);
            newY = MathHelper.func_76128_c(newMove.field_72448_b);
            newZ = MathHelper.func_76128_c(newMove.field_72449_c);
        }

        @NotNull final BlockPos start = AbstractPathJob.prepareStart(entity);
        @NotNull final BlockPos dest = new BlockPos(newX, newY, newZ);

        return setPathJob(
          new PathJobMoveToLocation(entity.field_70170_p, start, dest, (int) func_111269_d()),
          dest, speed);
    }

    public boolean isUnableToReachDestination()
    {
        return pathResult != null && pathResult.failedToReachDestination();
    }

    @Nullable
    private PathResult setPathJob(@NotNull final AbstractPathJob job, final BlockPos dest, final double speed)
    {
        func_75499_g();

        this.destination = dest;
        this.walkSpeed = speed;

        future = Pathfinding.enqueue(job);
        pathResult = job.getResult();
        return pathResult;
    }

    @Override
    public boolean func_75484_a(@NotNull Path path, final double speed)
    {
        final int pathLength = path.func_75874_d();
        Path tempPath = null;
        if (pathLength > 0 && !(path.func_75877_a(0) instanceof PathPointExtended))
        {
            //  Fix vanilla PathPoints to be PathPointExtended
            @NotNull final PathPointExtended[] newPoints = new PathPointExtended[pathLength];

            for (int i = 0; i < pathLength; ++i)
            {
                final PathPoint point = path.func_75877_a(i);
                newPoints[i] = new PathPointExtended(new BlockPos(point.field_75839_a, point.field_75837_b, point.field_75838_c));
            }

            tempPath = new Path(newPoints);

            final PathPointExtended finalPoint = newPoints[pathLength - 1];
            destination = new BlockPos(finalPoint.field_75839_a, finalPoint.field_75837_b, finalPoint.field_75838_c);
        }

        return super.func_75484_a(tempPath == null ? path : tempPath, speed);
    }

    @Override
    public void func_75501_e()
    {
        if (future != null)
        {
            if (!future.isDone())
            {
                return;
            }

            try
            {
                if (future.get() == null)
                {
                    future = null;
                    return;
                }

                func_75484_a(future.get(), walkSpeed);

                pathResult.setPathLength(func_75505_d().func_75874_d());
                pathResult.setStatus(PathResult.Status.IN_PROGRESS_FOLLOWING);

                final PathPoint p = func_75505_d().func_75870_c();
                if (p != null && destination == null)
                {
                    destination = new BlockPos(p.field_75839_a, p.field_75837_b, p.field_75838_c);

                    //  AbstractPathJob with no destination, did reach it's destination
                    pathResult.setPathReachesDestination(true);
                }
            }
            catch (@NotNull InterruptedException | ExecutionException e)
            {
                Log.getLogger().catching(e);
            }

            future = null;
        }

        int oldIndex = this.func_75500_f() ? 0 : this.func_75505_d().func_75873_e();
        super.func_75501_e();

        //  Ladder Workaround
        if (!this.func_75500_f())
        {
            @NotNull final PathPointExtended pEx = (PathPointExtended) this.func_75505_d().func_75877_a(this.func_75505_d().func_75873_e());

            if (pEx.isOnLadder())
            {
                final Vec3d vec3 = this.func_75505_d().func_75878_a(this.entity);

                if (vec3.func_186679_c(entity.field_70165_t, vec3.field_72448_b, entity.field_70161_v) < 0.1)
                {
                    //This way he is less nervous and gets up the ladder
                    double newSpeed = 0.05;
                    switch (pEx.getLadderFacing())
                    {
                        //  Any of these values is climbing, so adjust our direction of travel towards the ladder
                        case NORTH:
                            vec3.func_72441_c(0, 0, 1);
                            break;
                        case SOUTH:
                            vec3.func_72441_c(0, 0, -1);
                            break;
                        case WEST:
                            vec3.func_72441_c(1, 0, 0);
                            break;
                        case EAST:
                            vec3.func_72441_c(-1, 0, 0);
                            break;
                        //  Any other value is going down, so lets not move at all
                        default:
                            newSpeed = 0;
                            break;
                    }

                    this.entity.func_70605_aq().func_75642_a(vec3.field_72450_a, vec3.field_72448_b, vec3.field_72449_c, newSpeed);
                }
            }
            else if (entity.func_70090_H())
            {
                //  Prevent shortcuts when swimming
                final int curIndex = this.func_75505_d().func_75873_e();
                if (curIndex > 0
                      && (curIndex + 1) < this.func_75505_d().func_75874_d()
                      && this.func_75505_d().func_75877_a(curIndex - 1).field_75837_b != pEx.field_75837_b)
                {
                    //  Work around the initial 'spin back' when dropping into water
                    oldIndex = curIndex + 1;
                }

                this.func_75505_d().func_75872_c(oldIndex);

                Vec3d Vec3d = this.func_75505_d().func_75878_a(this.entity);

                if (Vec3d.func_72436_e(new Vec3d(entity.field_70165_t, Vec3d.field_72448_b, entity.field_70161_v)) < 0.1
                      && Math.abs(entity.field_70163_u - Vec3d.field_72448_b) < 0.5)
                {
                    this.func_75505_d().func_75872_c(this.func_75505_d().func_75873_e() + 1);
                    if (this.func_75500_f())
                    {
                        return;
                    }

                    Vec3d = this.func_75505_d().func_75878_a(this.entity);
                }

                this.entity.func_70605_aq().func_75642_a(Vec3d.field_72450_a, Vec3d.field_72448_b, Vec3d.field_72449_c, walkSpeed);
            }
            else
            {
                if (BlockUtils.isPathBlock(field_75513_b.func_180495_p(entity.func_180425_c().func_177977_b()).func_177230_c()))
                {
                    field_75511_d = 1.3;
                }
                else
                {
                    field_75511_d = 1.0;
                }
            }
        }

        if (pathResult != null && func_75500_f())
        {
            pathResult.setStatus(PathResult.Status.COMPLETE);
            pathResult = null;
        }
    }

    @Override
    protected void func_75508_h()
    {
        final int curNode = field_75514_c.func_75873_e();
        final int curNodeNext = curNode + 1;
        if (curNodeNext < field_75514_c.func_75874_d())
        {
            final PathPointExtended pEx = (PathPointExtended) field_75514_c.func_75877_a(curNode);
            final PathPointExtended pExNext = (PathPointExtended) field_75514_c.func_75877_a(curNodeNext);

            //  If current node is bottom of a ladder, then stay on this node until
            //  the entity reaches the bottom, otherwise they will try to head out early
            if (pEx.isOnLadder() && pEx.getLadderFacing() == EnumFacing.DOWN
                  && !pExNext.isOnLadder())
            {
                final Vec3d vec3 = func_75502_i();
                if ((vec3.field_72448_b - (double) pEx.field_75837_b) < 0.001)
                {
                    this.field_75514_c.func_75872_c(curNodeNext);
                }

                this.func_179677_a(vec3);
                return;
            }
        }

        super.func_75508_h();
    }

    /**
     * If null path or reached the end.
     */
    @Override
    public boolean func_75500_f()
    {
        return future == null && super.func_75500_f();
    }

    @Override
    public void func_75499_g()
    {
        if (future != null)
        {
            future.cancel(true);
            future = null;
        }

        if (pathResult != null)
        {
            pathResult.setStatus(PathResult.Status.CANCELLED);
            pathResult = null;
        }

        destination = null;
        super.func_75499_g();
    }

    /**
     * Used to find a tree.
     *
     * @param range in the range.
     * @param speed walking speed.
     * @return the result of the search.
     */
    public PathJobFindTree.TreePathResult moveToTree(final int range, final double speed)
    {
        @NotNull final BlockPos start = AbstractPathJob.prepareStart(entity);
        return (PathJobFindTree.TreePathResult) setPathJob(
          new PathJobFindTree(entity.field_70170_p, start, ((EntityCitizen) entity).getWorkBuilding().getLocation(), range), null, speed);
    }

    /**
     * Used to find a water.
     *
     * @param range in the range.
     * @param speed walking speed.
     * @param ponds a list of ponds.
     * @return the result of the search.
     */
    @Nullable
    public PathJobFindWater.WaterPathResult moveToWater(final int range, final double speed, final List<BlockPos> ponds)
    {
        @NotNull final BlockPos start = AbstractPathJob.prepareStart(entity);
        return (PathJobFindWater.WaterPathResult) setPathJob(
          new PathJobFindWater(entity.field_70170_p, start, ((EntityCitizen) entity).getWorkBuilding().getLocation(), range, ponds), null, speed);
    }

    /**
     * Used to move a living entity with a speed.
     *
     * @param e     the entity.
     * @param speed the speed.
     * @return the result.
     */
    @Nullable
    public PathResult moveToEntityLiving(@NotNull final Entity e, final double speed)
    {
        return moveToXYZ(e.field_70165_t, e.field_70163_u, e.field_70161_v, speed);
    }

    /**
     * Used to path away from a entity.
     *
     * @param e        the entity.
     * @param distance the distance to move to.
     * @param speed    the speed to run at.
     * @return the result of the pathing.
     */
    @Nullable
    public PathResult moveAwayFromEntityLiving(@NotNull final Entity e, final double distance, final double speed)
    {
        return moveAwayFromXYZ(e.func_180425_c(), distance, speed);
    }

    /**
     * Used to path away from a position.
     *
     * @param avoid the position to avoid.
     * @param range the range he should move out of.
     * @param speed the speed to run at.
     * @return the result of the pathing.
     */
    @Nullable
    public PathResult moveAwayFromXYZ(final BlockPos avoid, final double range, final double speed)
    {
        @NotNull final BlockPos start = AbstractPathJob.prepareStart(entity);

        return setPathJob(
          new PathJobMoveAwayFromLocation(entity.field_70170_p, start, avoid, (int) range, (int) func_111269_d()),
          null, speed);
    }
}
