package com.minecolonies.coremod.entity.pathfinding;

import com.minecolonies.coremod.configuration.Configurations;
import com.minecolonies.coremod.util.Log;
import net.minecraft.client.Minecraft;
import net.minecraft.client.gui.FontRenderer;
import net.minecraft.client.renderer.Tessellator;
import net.minecraft.client.renderer.VertexBuffer;
import net.minecraft.client.renderer.entity.RenderManager;
import net.minecraft.client.renderer.vertex.DefaultVertexFormats;
import net.minecraft.entity.Entity;
import net.minecraft.pathfinding.Path;
import net.minecraft.util.ResourceLocation;
import net.minecraftforge.fml.relauncher.Side;
import net.minecraftforge.fml.relauncher.SideOnly;
import org.jetbrains.annotations.NotNull;
import org.lwjgl.opengl.GL11;

import java.util.ConcurrentModificationException;
import java.util.Set;
import java.util.concurrent.*;

/**
 * Static class the handles all the Pathfinding.
 */
public final class Pathfinding
{
    private static final BlockingQueue<Runnable> jobQueue = new LinkedBlockingDeque<>();
    private static final ResourceLocation        TEXTURE  = new ResourceLocation("textures/gui/widgets.png");
    private static final ThreadPoolExecutor executor;
    static
    {
        executor = new ThreadPoolExecutor(1, Configurations.pathfindingMaxThreadCount, 10, TimeUnit.SECONDS, jobQueue);
    }
    private Pathfinding()
    {
        //Hides default constructor.
    }

    /**
     * Add a job to the queue for processing.
     *
     * @param job PathJob
     * @return a Future containing the Path
     */
    public static Future<Path> enqueue(@NotNull final AbstractPathJob job)
    {
        return executor.submit(job);
    }

    /**
     * Render debugging information for the pathfinding system.
     *
     * @param frame entity movement weight.
     */
    @SideOnly(Side.CLIENT)
    public static void debugDraw(final double frame)
    {
        if (AbstractPathJob.lastDebugNodesNotVisited == null)
        {
            return;
        }

        final Entity entity = Minecraft.func_71410_x().func_175606_aa();
        final double dx = entity.field_70142_S + (entity.field_70165_t - entity.field_70142_S) * frame;
        final double dy = entity.field_70137_T + (entity.field_70163_u - entity.field_70137_T) * frame;
        final double dz = entity.field_70136_U + (entity.field_70161_v - entity.field_70136_U) * frame;

        GL11.glPushMatrix();
        GL11.glPushAttrib(GL11.GL_ALL_ATTRIB_BITS);
        GL11.glTranslated(-dx, -dy, -dz);

        GL11.glDisable(GL11.GL_TEXTURE_2D);
        GL11.glDisable(GL11.GL_BLEND);
        GL11.glDisable(GL11.GL_LIGHTING);
        GL11.glEnable(GL11.GL_CULL_FACE);
        GL11.glEnable(GL11.GL_DEPTH_TEST);

        final Set<Node> debugNodesNotVisited;
        final Set<Node> debugNodesVisited;
        final Set<Node> debugNodesPath;

        synchronized (AbstractPathJob.debugNodeMonitor)
        {
            debugNodesNotVisited = AbstractPathJob.lastDebugNodesNotVisited;
            debugNodesVisited = AbstractPathJob.lastDebugNodesVisited;
            debugNodesPath = AbstractPathJob.lastDebugNodesPath;
        }

        try
        {
            for (@NotNull final Node n : debugNodesNotVisited)
            {
                debugDrawNode(n, (byte) 255, (byte) 0, (byte) 0);
            }

            for (@NotNull final Node n : debugNodesVisited)
            {
                debugDrawNode(n, (byte) 0, (byte) 0, (byte) 255);
            }

            if (debugNodesPath != null)
            {
                for (@NotNull final Node n : debugNodesPath)
                {
                    debugDrawNode(n, (byte) 0, (byte) 255, (byte) 0);
                }
            }
        }
        catch (final ConcurrentModificationException exc)
        {
            Log.getLogger().catching(exc);
        }

        GL11.glPopAttrib();
        GL11.glPopMatrix();
    }

    @SideOnly(Side.CLIENT)
    private static void debugDrawNode(@NotNull final Node n, final byte r, final byte g, final byte b)
    {
        GL11.glPushMatrix();
        GL11.glTranslated((double) n.pos.func_177958_n() + 0.375, (double) n.pos.func_177956_o() + 0.375, (double) n.pos.func_177952_p() + 0.375);

        final float f = 1.6F;
        final float f1 = (float) (0.016666668D * f / 2);

        //  Nameplate

        final Entity entity = Minecraft.func_71410_x().func_175606_aa();
        final double dx = n.pos.func_177958_n() - entity.field_70165_t;
        final double dy = n.pos.func_177956_o() - entity.field_70163_u;
        final double dz = n.pos.func_177952_p() - entity.field_70161_v;
        if (Math.sqrt(dx * dx + dy * dy + dz * dz) <= 5D)
        {
            renderDebugText(n, f1);
        }

        GL11.glScaled(0.25, 0.25, 0.25);

        GL11.glBegin(GL11.GL_QUADS);

        GL11.glColor3ub(r, g, b);

        //  X+ Facing
        GL11.glVertex3d(1.0, 0.0, 0.0);
        GL11.glVertex3d(1.0, 1.0, 0.0);
        GL11.glVertex3d(1.0, 1.0, 1.0);
        GL11.glVertex3d(1.0, 0.0, 1.0);

        //  X- Facing
        GL11.glVertex3d(0.0, 0.0, 1.0);
        GL11.glVertex3d(0.0, 1.0, 1.0);
        GL11.glVertex3d(0.0, 1.0, 0.0);
        GL11.glVertex3d(0.0, 0.0, 0.0);

        //  Z-
        GL11.glVertex3d(0.0, 0.0, 0.0);
        GL11.glVertex3d(0.0, 1.0, 0.0);
        GL11.glVertex3d(1.0, 1.0, 0.0);
        GL11.glVertex3d(1.0, 0.0, 0.0);

        //  Z+
        GL11.glVertex3d(1.0, 0.0, 1.0);
        GL11.glVertex3d(1.0, 1.0, 1.0);
        GL11.glVertex3d(0.0, 1.0, 1.0);
        GL11.glVertex3d(0.0, 0.0, 1.0);

        //  Y+
        GL11.glVertex3d(1.0, 1.0, 1.0);
        GL11.glVertex3d(1.0, 1.0, 0.0);
        GL11.glVertex3d(0.0, 1.0, 0.0);
        GL11.glVertex3d(0.0, 1.0, 1.0);

        //  Y-
        GL11.glVertex3d(0.0, 0.0, 1.0);
        GL11.glVertex3d(0.0, 0.0, 0.0);
        GL11.glVertex3d(1.0, 0.0, 0.0);
        GL11.glVertex3d(1.0, 0.0, 1.0);

        GL11.glEnd();

        if (n.parent != null)
        {
            GL11.glBegin(GL11.GL_LINES);
            GL11.glColor3f(0.75F, 0.75F, 0.75F);

            final double pdx = n.parent.pos.func_177958_n() - n.pos.func_177958_n() + 0.125;
            final double pdy = n.parent.pos.func_177956_o() - n.pos.func_177956_o() + 0.125;
            final double pdz = n.parent.pos.func_177952_p() - n.pos.func_177952_p() + 0.125;

            GL11.glVertex3d(0.5, 0.5, 0.5);
            GL11.glVertex3d(pdx / 0.25, pdy / 0.25, pdz / 0.25);

            GL11.glEnd();
        }

        GL11.glPopMatrix();
    }

    @SideOnly(Side.CLIENT)
    private static void renderDebugText(@NotNull final Node n, final float f1)
    {
        final String s1 = String.format("F: %.3f [%d]", n.getCost(), n.getCounterAdded());
        final String s2 = String.format("G: %.3f [%d]", n.getScore(), n.getCounterVisited());
        final FontRenderer fontrenderer = Minecraft.func_71410_x().field_71466_p;
        GL11.glPushAttrib(GL11.GL_ALL_ATTRIB_BITS);
        GL11.glPushMatrix();
        GL11.glTranslatef(0.0F, 0.75F, 0.0F);
        GL11.glNormal3f(0.0F, 1.0F, 0.0F);

        final RenderManager renderManager = Minecraft.func_71410_x().func_175598_ae();
        GL11.glRotatef(-renderManager.field_78735_i, 0.0F, 1.0F, 0.0F);
        GL11.glRotatef(renderManager.field_78732_j, 1.0F, 0.0F, 0.0F);
        GL11.glScalef(-f1, -f1, f1);
        GL11.glTranslatef(0.0F, (float) (0.25D / f1), 0.0F);
        GL11.glDepthMask(false);

        final Tessellator tessellator = Tessellator.func_178181_a();
        final VertexBuffer VertexBuffer = tessellator.func_178180_c();
        GL11.glDisable(GL11.GL_TEXTURE_2D);

        VertexBuffer.func_181668_a(GL11.GL_QUADS, DefaultVertexFormats.field_181707_g);
        final int i = Math.max(fontrenderer.func_78256_a(s1), fontrenderer.func_78256_a(s2)) / 2;

        //that should set the colors correctly
        VertexBuffer.func_181662_b((double) (-i - 1), -5.0D, 0.0D).func_181666_a(0.0F, 0.0F, 0.0F, 0.25F).func_181675_d();
        VertexBuffer.func_181662_b((double) (-i - 1), 12.0D, 0.0D).func_181666_a(0.0F, 0.0F, 0.0F, 0.25F).func_181675_d();
        VertexBuffer.func_181662_b((double) (i + 1), 12.0D, 0.0D).func_181666_a(0.0F, 0.0F, 0.0F, 0.25F).func_181675_d();
        VertexBuffer.func_181662_b((double) (i + 1), -5.0D, 0.0D).func_181666_a(0.0F, 0.0F, 0.0F, 0.25F).func_181675_d();
        tessellator.func_78381_a();
        GL11.glEnable(GL11.GL_TEXTURE_2D);
        GL11.glEnable(GL11.GL_BLEND);
        Minecraft.func_71410_x().field_71446_o.func_110577_a(TEXTURE);
        GL11.glDepthMask(true);
        GL11.glTranslatef(0.0F, -5F, 0.0F);
        fontrenderer.func_78276_b(s1, -fontrenderer.func_78256_a(s1) / 2, 0, 553648127);
        GL11.glTranslatef(0.0F, 8F, 0.0F);
        fontrenderer.func_78276_b(s2, -fontrenderer.func_78256_a(s2) / 2, 0, 553648127);
        GL11.glPopMatrix();
        GL11.glPopAttrib();
    }
}
