package com.minecolonies.coremod.entity.pathfinding;

import com.minecolonies.coremod.colony.buildings.AbstractBuildingWorker;
import com.minecolonies.coremod.colony.buildings.BuildingMiner;
import com.minecolonies.coremod.colony.jobs.JobBuilder;
import com.minecolonies.coremod.colony.jobs.JobMiner;
import com.minecolonies.coremod.entity.EntityCitizen;
import com.minecolonies.coremod.entity.ai.citizen.miner.Level;
import com.minecolonies.coremod.entity.ai.citizen.miner.Node;
import com.minecolonies.coremod.util.BlockPosUtil;
import com.minecolonies.coremod.util.EntityUtils;
import net.minecraft.util.math.BlockPos;
import org.jetbrains.annotations.NotNull;

import java.awt.geom.Point2D;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * Proxy handling walkToX tasks.
 */
public class WalkToProxy
{
    /**
     * The distance the worker can path directly without the proxy.
     */
    private static final int MIN_RANGE_FOR_DIRECT_PATH = 400;

    /**
     * The min distance a worker has to have to a proxy.
     */
    private static final int MIN_DISTANCE = 25;

    /**
     * Lead the miner to the other side of the shaft.
     */
    private static final int OTHER_SIDE_OF_SHAFT = 6;

    /**
     * The worker entity associated with the proxy.
     */
    private final EntityCitizen worker;

    /**
     * The current proxy the citizen paths to.
     */
    private BlockPos currentProxy;

    /**
     * List of proxies the worker has to follow.
     */
    private ArrayList<BlockPos> proxyList = new ArrayList<>();

    /**
     * Current target the worker has.
     */
    private BlockPos target;

    /**
     * Creates a walkToProxy for a certain worker.
     *
     * @param worker the worker.
     */
    public WalkToProxy(EntityCitizen worker)
    {
        this.worker = worker;
    }

    /**
     * Leads the worker to a certain position due to proxies.
     *
     * @param target the position.
     * @param range  the range.
     * @return true if arrived.
     */
    public boolean walkToBlock(@NotNull BlockPos target, int range)
    {
        return walkToBlock(target, range, false);
    }

    /**
     * Take the direct path to a certain location.
     *
     * @param target the target position.
     * @param range  the range.
     * @param onMove worker on move or not?
     * @return true if arrived.
     */
    private boolean takeTheDirectPath(@NotNull BlockPos target, int range, boolean onMove)
    {
        if (onMove)
        {
            final int targetY = worker.getColonyJob() instanceof JobBuilder ? worker.func_180425_c().func_177956_o() : target.func_177956_o();
            return EntityUtils.isWorkerAtSiteWithMove(worker, target.func_177958_n(), target.func_177956_o(), target.func_177952_p(), range)
                     || EntityUtils.isWorkerAtSite(worker, target.func_177958_n(), targetY, target.func_177952_p(), range + 1);
        }
        else
        {
            return !EntityUtils.isWorkerAtSite(worker, target.func_177958_n(), target.func_177956_o(), target.func_177952_p(), range);
        }
    }

    /**
     * Leads the worker to a certain position due to proxies.
     *
     * @param target the target position.
     * @param range  the range.
     * @param onMove worker on move or not?
     * @return true if arrived.
     */
    public boolean walkToBlock(@NotNull BlockPos target, int range, boolean onMove)
    {
        if(!target.equals(this.target))
        {
            reset();
            this.target = target;
        }

        final double distanceToPath = worker.getColonyJob() instanceof JobBuilder
                ? BlockPosUtil.getDistanceSquared2D(worker.func_180425_c(), target) : BlockPosUtil.getDistanceSquared(worker.func_180425_c(), target);

        if (distanceToPath <= MIN_RANGE_FOR_DIRECT_PATH)
        {
            if (distanceToPath <= MIN_DISTANCE)
            {
                currentProxy = null;
            }
            else
            {
                currentProxy = target;
            }

            proxyList = new ArrayList<>();
            return takeTheDirectPath(target, range, onMove);
        }

        if (currentProxy == null)
        {
            currentProxy = fillProxyList(target, distanceToPath);
        }

        final double distanceToProxy = BlockPosUtil.getDistanceSquared2D(worker.func_180425_c(), currentProxy);
        final double distanceToNextProxy = proxyList.isEmpty() ? BlockPosUtil.getDistanceSquared2D(worker.func_180425_c(), target)
                                             : BlockPosUtil.getDistanceSquared2D(worker.func_180425_c(), proxyList.get(0));
        final double distanceProxyNextProxy = proxyList.isEmpty() ? BlockPosUtil.getDistanceSquared2D(currentProxy, target)
                                                : BlockPosUtil.getDistanceSquared2D(currentProxy, proxyList.get(0));
        if (distanceToProxy < MIN_DISTANCE || distanceToNextProxy < distanceProxyNextProxy)
        {
            if (proxyList.isEmpty())
            {
                currentProxy = target;
            }

            if (proxyList.isEmpty())
            {
                return takeTheDirectPath(target, range, onMove);
            }

            worker.func_70661_as().func_75499_g();
            currentProxy = proxyList.get(0);
            proxyList.remove(0);
        }

        if (currentProxy != null && !EntityUtils.isWorkerAtSiteWithMove(worker, currentProxy.func_177958_n(), currentProxy.func_177956_o(), currentProxy.func_177952_p(), range))
        {
            //only walk to the block
            return !onMove;
        }

        return !onMove;
    }

    /**
     * Calculates a list of proxies to a certain target for a worker.
     *
     * @param target         the target.
     * @param distanceToPath the complete distance.
     * @return the first position to path to.
     */
    @NotNull
    private BlockPos fillProxyList(@NotNull BlockPos target, double distanceToPath)
    {
        BlockPos proxyPoint;

        final AbstractBuildingWorker building = worker.getWorkBuilding();
        if (worker.getColonyJob() != null && worker.getColonyJob() instanceof JobMiner && building instanceof BuildingMiner)
        {
            proxyPoint = getMinerProxy(target, distanceToPath, (BuildingMiner) building);
        }
        else
        {
            proxyPoint = getProxy(target, worker.func_180425_c(), distanceToPath);
        }

        if (!proxyList.isEmpty())
        {
            proxyList.remove(0);
        }

        return proxyPoint;
    }

    /**
     * Reset the proxy.
     */
    public void reset()
    {
        currentProxy = null;
        proxyList = new ArrayList<>();
    }

    /**
     * Returns a proxy point to the goal for the miner especially.
     *
     * @param target         the target.
     * @param distanceToPath the total distance.
     * @return a proxy or, if not applicable null.
     */
    @NotNull
    private BlockPos getMinerProxy(final BlockPos target, final double distanceToPath, @NotNull final BuildingMiner building)
    {
        final Level level = building.getCurrentLevel();
        final BlockPos ladderPos = building.getLadderLocation();

        //If his current working level is null, we have nothing to worry about.
        if (level != null)
        {
            final int levelDepth = level.getDepth() + 2;
            final int targetY = target.func_177956_o();
            final int workerY = worker.func_180425_c().func_177956_o();

            //Check if miner is underground in shaft and his target is overground.
            if (workerY <= levelDepth && targetY > levelDepth)
            {
                if(level.getRandomNode() != null && level.getRandomNode().getParent() != null)
                {
                    com.minecolonies.coremod.entity.ai.citizen.miner.Node currentNode = level.getNode(level.getRandomNode().getParent());
                    while (new Point2D.Double(currentNode.getX(), currentNode.getZ()) != currentNode.getParent() && currentNode.getParent() != null)
                    {
                        proxyList.add(new BlockPos(currentNode.getX(), levelDepth, currentNode.getZ()));
                        currentNode = level.getNode(currentNode.getParent());
                    }
                }

                proxyList.add(
                        new BlockPos(
                                ladderPos.func_177958_n() + building.getVectorX() * OTHER_SIDE_OF_SHAFT,
                                level.getDepth(),
                                ladderPos.func_177952_p() + building.getVectorZ() * OTHER_SIDE_OF_SHAFT));
                return getProxy(target, worker.func_180425_c(), distanceToPath);

                //If he already is at ladder location, the closest node automatically will be his hut block.
            }
            //Check if target is underground in shaft and miner is over it.
            else if (targetY <= levelDepth && workerY > levelDepth)
            {
                final BlockPos buildingPos = building.getLocation();
                BlockPos newProxy;

                //First calculate way to miner building.
                newProxy = getProxy(buildingPos, worker.func_180425_c(), BlockPosUtil.getDistanceSquared(worker.func_180425_c(), buildingPos));


                //Then add the ladder position as the latest node.
                proxyList.add(
                  new BlockPos(
                                ladderPos.func_177958_n() + building.getVectorX() * OTHER_SIDE_OF_SHAFT,
                          level.getDepth(),
                          ladderPos.func_177952_p() + building.getVectorZ() * OTHER_SIDE_OF_SHAFT));

                if(level.getRandomNode() != null && level.getRandomNode().getParent() != null)
                {
                    final List<BlockPos> nodesToTarget = new ArrayList<>();
                    com.minecolonies.coremod.entity.ai.citizen.miner.Node currentNode = level.getNode(level.getRandomNode().getParent());
                    while (new Point2D.Double(currentNode.getX(), currentNode.getZ()) != currentNode.getParent() && currentNode.getParent() != null)
                    {
                        nodesToTarget.add(new BlockPos(currentNode.getX(), levelDepth, currentNode.getZ()));
                        currentNode = level.getNode(currentNode.getParent());
                    }

                    for (int i = nodesToTarget.size() - 1; i >= 0; i--)
                    {
                        proxyList.add(nodesToTarget.get(i));
                    }
                }

                return newProxy;
            }
            //If he is on the same Y level as his target and both underground.
            else if (targetY <= levelDepth)
            {
                double closestNode = Double.MAX_VALUE;
                Node lastNode = null;
                for(final Map.Entry<Point2D, Node> node : level.getNodes().entrySet())
                {
                    final double distanceToNode = node.getKey().distance(worker.func_180425_c().func_177958_n(), worker.func_180425_c().func_177952_p());
                    if(distanceToNode < closestNode)
                    {
                        lastNode = node.getValue();
                        closestNode = distanceToNode;
                    }
                }

                if(lastNode != null && lastNode.getParent() != null)
                {
                    com.minecolonies.coremod.entity.ai.citizen.miner.Node currentNode = level.getNode(lastNode.getParent());
                    while (new Point2D.Double(currentNode.getX(), currentNode.getZ()) != currentNode.getParent() && currentNode.getParent() != null)
                    {
                        proxyList.add(new BlockPos(currentNode.getX(), levelDepth, currentNode.getZ()));
                        currentNode = level.getNode(currentNode.getParent());
                    }
                }

                if(level.getRandomNode().getParent() != null)
                {
                    final List<BlockPos> nodesToTarget = new ArrayList<>();
                    com.minecolonies.coremod.entity.ai.citizen.miner.Node currentNode = level.getNode(level.getRandomNode().getParent());
                    while (new Point2D.Double(currentNode.getX(), currentNode.getZ()) != currentNode.getParent() && currentNode.getParent() != null)
                    {
                        nodesToTarget.add(new BlockPos(currentNode.getX(), levelDepth, currentNode.getZ()));
                        currentNode = level.getNode(currentNode.getParent());
                    }

                    for (int i = nodesToTarget.size() - 1; i >= 0; i--)
                    {
                        proxyList.add(nodesToTarget.get(i));
                    }
                }

                if(!proxyList.isEmpty())
                {
                    return proxyList.get(0);
                }
                return target;
            }
        }

        return getProxy(target, worker.func_180425_c(), distanceToPath);
    }

    /**
     * Returns a proxy point to the goal.
     *
     * @param target         the target.
     * @param distanceToPath the total distance.
     * @return a proxy or, if not applicable null.
     */
    @NotNull
    private BlockPos getProxy(@NotNull BlockPos target, @NotNull BlockPos position, double distanceToPath)
    {
        if (worker.getColony() == null)
        {
            return target;
        }

        double weight = Double.MAX_VALUE;
        BlockPos proxyPoint = null;

        for (BlockPos wayPoint : worker.getColony().getWayPoints(position, target))
        {
            final double simpleDistance = BlockPosUtil.getDistanceSquared(position, wayPoint);
            final double currentWeight = simpleDistance * simpleDistance + BlockPosUtil.getDistanceSquared(wayPoint, target);
            if (currentWeight < weight
                  && BlockPosUtil.getDistanceSquared2D(wayPoint, target) < distanceToPath
                  && simpleDistance > MIN_DISTANCE
                  && simpleDistance < distanceToPath
                  && !proxyList.contains(proxyPoint))
            {
                proxyPoint = wayPoint;
                weight = currentWeight;
            }
        }

        if (proxyList.contains(proxyPoint))
        {
            proxyPoint = null;
        }

        if (proxyPoint != null)
        {
            proxyList.add(proxyPoint);

            getProxy(target, proxyPoint, distanceToPath);

            return proxyList.get(0);
        }

        //No proxy point exists.
        return target;
    }
}
