package com.minecolonies.coremod.inventory;

import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.inventory.IInventory;
import net.minecraft.item.ItemSeeds;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.nbt.NBTTagList;
import net.minecraft.util.text.ITextComponent;
import net.minecraft.util.text.TextComponentTranslation;
import net.minecraftforge.common.util.Constants;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

/**
 * The custom chest of the field.
 */
public class InventoryField implements IInventory
{
    /**
     * NBTTag to store the slot.
     */
    private static final String TAG_SLOT = "slot";

    /**
     * NBTTag to store the items.
     */
    private static final String TAG_ITEMS = "items";

    /**
     * NBTTag to store the custom name.
     */
    private static final String TAG_CUSTOM_NAME = "name";

    /**
     * NBTTag to store the inventory.
     */
    private static final String TAG_INVENTORY = "inventory";

    /**
     * Returned slot if no slat has been found.
     */
    private static final int NO_SLOT = -1;

    /**
     * The inventory stack.
     */
    @NotNull
    private ItemStack[] stackResult = new ItemStack[1];

    /**
     * The custom name of the inventory.
     */
    private String customName = "";

    /**
     * Updated after the inventory has been changed.
     */
    private boolean inventoryChanged = false;

    /**
     * Creates the inventory of the citizen.
     *
     * @param title Title of the inventory.
     */
    public InventoryField(final String title)
    {
        super();
        this.customName = title;
    }

    /**
     * Checks if the inventory has been changed and then resets the boolean.
     *
     * @return true if it changed.
     */
    public boolean hasInventoryChanged()
    {
        if (inventoryChanged)
        {
            inventoryChanged = false;
            return true;
        }
        return false;
    }

    /**
     * Used to retrieve variables.
     *
     * @param compound with the give tag.
     */
    public void readFromNBT(@NotNull final NBTTagCompound compound)
    {
        final NBTTagList nbttaglist = compound.func_150295_c(TAG_ITEMS, Constants.NBT.TAG_COMPOUND);
        this.stackResult = new ItemStack[this.func_70302_i_()];

        for (int i = 0; i < nbttaglist.func_74745_c(); ++i)
        {
            final NBTTagCompound nbttagcompound = nbttaglist.func_150305_b(i);
            final int j = nbttagcompound.func_74771_c(TAG_SLOT) & Byte.MAX_VALUE;

            if (j != NO_SLOT && j < this.stackResult.length)
            {
                this.stackResult[j] = ItemStack.func_77949_a(nbttagcompound);
            }
        }

        if (compound.func_150297_b(TAG_CUSTOM_NAME, Constants.NBT.TAG_STRING))
        {
            this.customName = compound.func_74779_i(TAG_CUSTOM_NAME);
        }
    }

    @Override
    public int func_70302_i_()
    {
        return 1;
    }

    /**
     * Getter for the stack in the inventory. Since there is only one slot return always that one.
     *
     * @param index the slot.
     * @return the itemStack in it.
     */
    @Override
    public ItemStack func_70301_a(final int index)
    {
        return this.stackResult[0];
    }

    @Nullable
    @Override
    public ItemStack func_70298_a(final int index, final int count)
    {
        if (this.stackResult[index] == null)
        {
            return null;
        }

        if (this.stackResult[index].field_77994_a <= count)
        {
            final ItemStack itemStack1 = this.stackResult[index];
            this.stackResult[index] = null;
            this.func_70296_d();
            return itemStack1;
        }
        else
        {
            @NotNull final ItemStack itemstack = this.stackResult[index].func_77979_a(count);

            if (this.stackResult[index].field_77994_a == 0)
            {
                this.stackResult[index] = null;
            }

            this.func_70296_d();
            return itemstack;
        }
    }

    @Nullable
    @Override
    public ItemStack func_70304_b(final int index)
    {
        if (this.stackResult[index] == null)
        {
            return null;
        }

        final ItemStack itemstack = this.stackResult[index];
        this.stackResult[index] = null;
        return itemstack;
    }

    /**
     * Sets the given item stack to the specified slot in the inventory (can be crafting or armor sections).
     *
     * @param index the slot to set the itemStack.
     * @param stack the itemStack to set.
     */
    @Override
    public void func_70299_a(final int index, @Nullable final ItemStack stack)
    {
        this.stackResult[index] = stack;

        if (stack != null && stack.field_77994_a > this.func_70297_j_())
        {
            stack.field_77994_a = this.func_70297_j_();
        }

        this.func_70296_d();
    }

    @Override
    public int func_70297_j_()
    {
        return 1;
    }

    @Override
    public void func_70296_d()
    {
        this.inventoryChanged = true;
    }

    @Override
    public boolean func_70300_a(final EntityPlayer entityPlayer)
    {
        return true;
    }

    /**
     * Called when inventory is opened by a player.
     *
     * @param entityPlayer the player who opened the inventory.
     */
    @Override
    public void func_174889_b(final EntityPlayer entityPlayer)
    {
        /*
         * This may be filled in order to specify some custom handling.
         */
    }

    /**
     * Called after the inventory has been closed by a player.
     *
     * @param entityPlayer the player who opened the inventory.
     */
    @Override
    public void func_174886_c(final EntityPlayer entityPlayer)
    {
        /*
         * This may be filled in order to specify some custom handling.
         */
    }

    @Override
    public boolean func_94041_b(final int index, @Nullable final ItemStack itemStack)
    {
        return index == 0 && itemStack != null && itemStack.func_77973_b() instanceof ItemSeeds;
    }

    /**
     * This may be used in order to return values of different GUI areas like the ones in the beacon.
     *
     * @param id the id of the field.
     * @return the value of the field.
     */
    @Override
    public int func_174887_a_(final int id)
    {
        return 0;
    }

    /**
     * This may be used to set GUI areas with a certain id and value.
     *
     * @param id    some id.
     * @param value some value.
     */
    @Override
    public void func_174885_b(final int id, final int value)
    {
        /*
         * We currently need no fields.
         */
    }

    /**
     * Returns the number of fields.
     *
     * @return the amount.
     */
    @Override
    public int func_174890_g()
    {
        return 0;
    }

    /**
     * Completely clears the inventory.
     */
    @Override
    public void func_174888_l()
    {
        for (int i = 0; i < this.stackResult.length; ++i)
        {
            this.stackResult[i] = null;
        }
    }

    /**
     * Used to store variables.
     *
     * @param compound with the given tag.
     */
    public void writeToNBT(@NotNull final NBTTagCompound compound)
    {
        @NotNull final NBTTagList nbttaglist = new NBTTagList();

        for (int i = 0; i < this.stackResult.length; ++i)
        {
            if (this.stackResult[i] != null)
            {
                @NotNull final NBTTagCompound nbttagcompound = new NBTTagCompound();
                nbttagcompound.func_74774_a(TAG_SLOT, (byte) i);
                this.stackResult[i].func_77955_b(nbttagcompound);
                nbttaglist.func_74742_a(nbttagcompound);
            }
        }

        compound.func_74782_a(TAG_ITEMS, nbttaglist);

        if (this.func_145818_k_())
        {
            compound.func_74778_a(TAG_CUSTOM_NAME, this.customName);
        }

        compound.func_74782_a(TAG_INVENTORY, nbttaglist);
    }

    /**
     * Setter of the customName of the inventory.
     *
     * @param customName the name to set.
     */
    public void setCustomName(final String customName)
    {
        this.customName = customName;
    }

    @Override
    public boolean func_145818_k_()
    {
        return true;
    }

    @NotNull
    @Override
    public ITextComponent func_145748_c_()
    {
        return new TextComponentTranslation(this.func_70005_c_());
    }

    /**
     * Get the name of this object. For citizens this returns their name.
     *
     * @return the name of the inventory.
     */
    @NotNull
    @Override
    public String func_70005_c_()
    {
        return this.func_145818_k_() ? this.customName : "field.inventory";
    }
}
