package com.minecolonies.coremod.tileentities;

import com.minecolonies.coremod.colony.Colony;
import com.minecolonies.coremod.colony.ColonyManager;
import com.minecolonies.coremod.inventory.InventoryField;
import com.minecolonies.coremod.util.EntityUtils;
import com.minecolonies.coremod.util.LanguageHandler;
import net.minecraft.entity.Entity;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.network.NetworkManager;
import net.minecraft.network.play.server.SPacketUpdateTileEntity;
import net.minecraft.tileentity.TileEntityChest;
import net.minecraft.world.World;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Random;

/**
 * The scarecrow tile entity to store extra data.
 */
public class ScarecrowTileEntity extends TileEntityChest
{
    /**
     * NBTTag to store the type.
     */
    private static final String TAG_TYPE = "type";

    /**
     * NBTag to store the name.
     */
    private static final String TAG_NAME = "name";

    /**
     * Random generator.
     */
    private final Random random = new Random();

    /**
     * The inventory connected with the scarecrow.
     */
    private InventoryField inventoryField;

    /**
     * The type of the scarecrow.
     */
    private ScareCrowType type;

    /**
     * Name of the scarecrow, string set in the GUI.
     */
    private String name = LanguageHandler.format("com.minecolonies.coremod.gui.scarecrow.user", LanguageHandler.format("com.minecolonies.coremod.gui.scarecrow.user.noone"));

    /**
     * Creates an instance of the tileEntity.
     */
    public ScarecrowTileEntity()
    {
        super();
        this.inventoryField = new InventoryField(name);
    }

    /**
     * Getter of the name of the tileEntity.
     *
     * @return the string.
     */
    public String getDesc()
    {
        return name;
    }

    /**
     * Setter for the name.
     *
     * @param name string to set.
     */
    public void setName(final String name)
    {
        this.name = name;
    }

    ///////////---- Following methods are used to update the tileEntity between client and server ----///////////

    @NotNull
    @Override
    public SPacketUpdateTileEntity func_189518_D_()
    {
        @NotNull final NBTTagCompound tag = new NBTTagCompound();
        func_189515_b(tag);
        return new SPacketUpdateTileEntity(this.func_174877_v(), 0, tag);
    }

    @Override
    public void onDataPacket(final NetworkManager net, @NotNull final SPacketUpdateTileEntity pkt)
    {
        func_145839_a(pkt.func_148857_g());
    }

    /////////////--------------------------- End Synchronization-area ---------------------------- /////////////

    @Override
    public void onLoad()
    {
        super.onLoad();
        final World world = func_145831_w();

        @Nullable final Colony colony = ColonyManager.getColony(world, field_174879_c);
        if (colony != null && colony.getField(field_174879_c) == null)
        {
            @Nullable final Entity entity = EntityUtils.getEntityFromUUID(world, colony.getPermissions().getOwner());

            if (entity instanceof EntityPlayer)
            {
                colony.addNewField(this, ((EntityPlayer) entity).field_71071_by, field_174879_c, world);
            }
        }
    }

    @Override
    public void func_145839_a(final NBTTagCompound compound)
    {
        super.func_145839_a(compound);

        type = ScareCrowType.values()[compound.func_74762_e(TAG_TYPE)];
        getInventoryField().readFromNBT(compound);
        name = compound.func_74779_i(TAG_NAME);
    }

    @Override
    public NBTTagCompound func_189515_b(final NBTTagCompound compound)
    {
        super.func_189515_b(compound);

        compound.func_74768_a(TAG_TYPE, this.getType().ordinal());
        getInventoryField().writeToNBT(compound);
        compound.func_74778_a(TAG_NAME, name);
        return compound;
    }

    /**
     * Returns the type of the scarecrow (Important for the rendering).
     *
     * @return the enum type.
     */
    public ScareCrowType getType()
    {
        if (this.type == null)
        {
            this.type = ScareCrowType.values()[this.random.nextInt(2)];
        }
        return this.type;
    }

    /**
     * Get the inventory connected with the scarecrow.
     *
     * @return the inventory field of this scarecrow
     */
    public InventoryField getInventoryField()
    {
        return inventoryField;
    }

    /**
     * Set the inventory connected with the scarecrow.
     *
     * @param inventoryField the field to set it to
     */
    public final void setInventoryField(final InventoryField inventoryField)
    {
        this.inventoryField = inventoryField;
    }

    /**
     * Enum describing the different textures the scarecrow has.
     */
    public enum ScareCrowType
    {
        PUMPKINHEAD,
        NORMAL
    }
}
