package com.minecolonies.coremod.tileentities;

import com.minecolonies.coremod.colony.Colony;
import com.minecolonies.coremod.colony.ColonyManager;
import com.minecolonies.coremod.colony.ColonyView;
import com.minecolonies.coremod.colony.buildings.AbstractBuilding;
import com.minecolonies.coremod.colony.permissions.Permissions;
import com.minecolonies.coremod.util.Log;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.network.NetworkManager;
import net.minecraft.network.play.server.SPacketUpdateTileEntity;
import net.minecraft.tileentity.TileEntityChest;
import net.minecraft.util.math.BlockPos;
import org.jetbrains.annotations.NotNull;

/**
 * Class which handles the tileEntity of our colonyBuildings.
 */
public class TileEntityColonyBuilding extends TileEntityChest
{
    /**
     * NBTTag to store the colony id.
     */
    private static final String TAG_COLONY = "colony";
    private static final String TAG_MIRROR = "mirror";

    /**
     * The colony id.
     */
    private int colonyId = 0;

    /**
     * The colony.
     */
    private Colony colony;

    /**
     * The building the tileEntity belongs to.
     */
    private AbstractBuilding building;

    /**
     * Check if the building has a mirror.
     */
    private boolean mirror;

    /**
     * Empty standard constructor.
     */
    public TileEntityColonyBuilding()
    {
        super();
    }

    @Override
    public SPacketUpdateTileEntity func_189518_D_()
    {
        final NBTTagCompound compound = new NBTTagCompound();
        compound.func_74768_a(TAG_COLONY, colonyId);
        return new SPacketUpdateTileEntity(this.getPosition(), 0, compound);
    }

    @NotNull
    @Override
    public NBTTagCompound func_189517_E_()
    {
        return func_189515_b(new NBTTagCompound());
    }

    @Override
    public void onDataPacket(final NetworkManager net, final SPacketUpdateTileEntity packet)
    {
        final NBTTagCompound compound = packet.func_148857_g();
        colonyId = compound.func_74762_e(TAG_COLONY);
    }

    @Override
    public void onChunkUnload()
    {
        if (building != null)
        {
            building.setTileEntity(null);
        }
    }

    /**
     * Returns the position of the tile entity.
     *
     * @return Block Coordinates of the tile entity.
     */
    public BlockPos getPosition()
    {
        return field_174879_c;
    }

    /**
     * Returns the colony ID.
     *
     * @return ID of the colony.
     */
    public int getColonyId()
    {
        return colonyId;
    }

    /**
     * Returns the colony of the tile entity.
     *
     * @return Colony of the tile entity.
     */
    public Colony getColony()
    {
        if (colony == null)
        {
            updateColonyReferences();
        }
        return colony;
    }

    /**
     * Synchronises colony references from the tile entity.
     */
    private void updateColonyReferences()
    {
        if (colony == null && field_145850_b != null)
        {
            if (colonyId == 0)
            {
                colony = ColonyManager.getColony(field_145850_b, this.func_174877_v());
            }
            else
            {
                colony = ColonyManager.getColony(colonyId);
            }

            if (colony == null)
            {
                //we tried to update the colony it is still missing... so we...
                if (field_145850_b.field_72995_K)
                {
                    /*
                     * It's most probably previewed building, please don't spam it here.
                     */
                }
                else
                {
                    //log on the server
                    Log.getLogger()
                      .warn(String.format("TileEntityColonyBuilding at %s:[%d,%d,%d] had colony.",
                        field_145850_b.func_72912_H().func_76065_j(), field_174879_c.func_177958_n(), field_174879_c.func_177956_o(), field_174879_c.func_177952_p()));
                }
            }
        }

        if (building == null && colony != null)
        {
            building = colony.getBuilding(getPosition());
            if (building != null && (field_145850_b == null || !field_145850_b.field_72995_K))
            {
                building.setTileEntity(this);
            }
        }
    }

    @Override
    public void func_73660_a()
    {
        super.func_73660_a();

        if (!field_145850_b.field_72995_K && colonyId == 0)
        {
            final Colony tempColony = ColonyManager.getColony(field_145850_b, this.getPosition());
            if (tempColony != null)
            {
                colonyId = tempColony.getID();
            }
        }
    }

    /**
     * Sets the colony of the tile entity.
     *
     * @param c Colony to set in references.
     */
    public void setColony(final Colony c)
    {
        colony = c;
        colonyId = c.getID();
        func_70296_d();
    }

    /**
     * Returns the building associated with the tile entity.
     *
     * @return {@link AbstractBuilding} associated with the tile entity.
     */
    public AbstractBuilding getBuilding()
    {
        if (building == null)
        {
            updateColonyReferences();
        }
        return building;
    }

    /**
     * Sets the building associated with the tile entity.
     *
     * @param b {@link AbstractBuilding} to associate with the tile entity.
     */
    public void setBuilding(final AbstractBuilding b)
    {
        building = b;
    }

    @Override
    public void func_70296_d()
    {
        super.func_70296_d();
        if (building!=null)
        {
            building.markDirty();
        }
    }


    /**
     * Returns the view of the building associated with the tile entity.
     *
     * @return {@link AbstractBuilding.View} the tile entity is associated with.
     */
    public AbstractBuilding.View getBuildingView()
    {
        final ColonyView c = ColonyManager.getColonyView(colonyId);
        return c == null ? null : c.getBuilding(getPosition());
    }



    @Override
    public void func_145839_a(final NBTTagCompound compound)
    {
        super.func_145839_a(compound);
        if (compound.func_74764_b(TAG_COLONY))
        {
            colonyId = compound.func_74762_e(TAG_COLONY);
        }

        updateColonyReferences();
        mirror = compound.func_74767_n(TAG_MIRROR);
    }

    @NotNull
    @Override
    public NBTTagCompound func_189515_b(@NotNull final NBTTagCompound compound)
    {
        super.func_189515_b(compound);
        /*
        if (colonyId == 0 && colony == null)
        {
            //todo: actually do something about it and not spam the server
        }
        */
        compound.func_74768_a(TAG_COLONY, colonyId);
        compound.func_74757_a(TAG_MIRROR, mirror);
        return compound;
    }

    @Override
    public boolean func_70300_a(@NotNull final EntityPlayer player)
    {
        return super.func_70300_a(player) && this.hasAccessPermission(player);
    }

    /**
     * Checks if the player has permission to access the hut.
     *
     * @param player Player to check permission of.
     * @return True when player has access, or building doesn't exist, otherwise false.
     */
    public boolean hasAccessPermission(final EntityPlayer player)
    {
        //TODO This is called every tick the GUI is open. Is that bad?
        return building == null || building.getColony().getPermissions().hasPermission(player, Permissions.Action.ACCESS_HUTS);
    }

    /**
     * Set if the entity is mirrored.
     * @param mirror true if so.
     */
    public void setMirror(final boolean mirror)
    {
        this.mirror = mirror;
    }

    /**
     * Check if building is mirrored.
     * @return true if so.
     */
    public boolean isMirrored()
    {
        return mirror;
    }
}
