package com.minecolonies.coremod.util;

import com.minecolonies.coremod.entity.EntityCitizen;
import com.minecolonies.coremod.entity.pathfinding.PathResult;
import io.netty.buffer.ByteBuf;
import net.minecraft.block.Block;
import net.minecraft.block.state.IBlockState;
import net.minecraft.command.ICommandSender;
import net.minecraft.entity.Entity;
import net.minecraft.entity.EntityLiving;
import net.minecraft.init.Blocks;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.nbt.NBTTagList;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.math.BlockPos;
import net.minecraft.util.math.MathHelper;
import net.minecraft.world.World;
import org.jetbrains.annotations.NotNull;

import java.util.List;

/**
 * Utility methods for BlockPos.
 */
public final class BlockPosUtil
{
    /**
     * Min distance to availate two positions as close.
     */
    private static final double CLOSE_DISTANCE = 4.84;

    private BlockPosUtil()
    {
        //Hide default constructor.
    }

    /**
     * Writes a Chunk Coordinate to an NBT compound, with a specific tag name.
     *
     * @param compound Compound to write to.
     * @param name     Name of the tag.
     * @param pos      Coordinates to write to NBT.
     */
    public static void writeToNBT(@NotNull final NBTTagCompound compound, final String name, @NotNull final BlockPos pos)
    {
        @NotNull final NBTTagCompound coordsCompound = new NBTTagCompound();
        coordsCompound.func_74768_a("x", pos.func_177958_n());
        coordsCompound.func_74768_a("y", pos.func_177956_o());
        coordsCompound.func_74768_a("z", pos.func_177952_p());
        compound.func_74782_a(name, coordsCompound);
    }

    /**
     * Reads Chunk Coordinates from an NBT Compound with a specific tag name.
     *
     * @param compound Compound to read data from.
     * @param name     Tag name to read data from.
     * @return Chunk coordinates read from the compound.
     */
    @NotNull
    public static BlockPos readFromNBT(@NotNull final NBTTagCompound compound, final String name)
    {
        final NBTTagCompound coordsCompound = compound.func_74775_l(name);
        final int x = coordsCompound.func_74762_e("x");
        final int y = coordsCompound.func_74762_e("y");
        final int z = coordsCompound.func_74762_e("z");
        return new BlockPos(x, y, z);
    }

    /**
     * Write a compound with chunk coordinate to a tag list.
     *
     * @param tagList Tag list to write compound with chunk coordinates to.
     * @param pos     Coordinate to write to the tag list.
     */
    public static void writeToNBTTagList(@NotNull final NBTTagList tagList, @NotNull final BlockPos pos)
    {
        @NotNull final NBTTagCompound coordsCompound = new NBTTagCompound();
        coordsCompound.func_74768_a("x", pos.func_177958_n());
        coordsCompound.func_74768_a("y", pos.func_177956_o());
        coordsCompound.func_74768_a("z", pos.func_177952_p());
        tagList.func_74742_a(coordsCompound);
    }

    /**
     * Reads a Chunk Coordinate from a tag list.
     *
     * @param tagList Tag list to read compound with chunk coordinate from.
     * @param index   Index in the tag list where the required chunk coordinate is.
     * @return Chunk coordinate read from the tag list.
     */
    @NotNull
    public static BlockPos readFromNBTTagList(@NotNull final NBTTagList tagList, final int index)
    {
        final NBTTagCompound coordsCompound = tagList.func_150305_b(index);
        final int x = coordsCompound.func_74762_e("x");
        final int y = coordsCompound.func_74762_e("y");
        final int z = coordsCompound.func_74762_e("z");
        return new BlockPos(x, y, z);
    }

    /**
     * Writes chunk coordinates to a {@link ByteBuf}.
     *
     * @param buf Buf to write to.
     * @param pos Coordinate to write.
     */
    public static void writeToByteBuf(@NotNull final ByteBuf buf, @NotNull final BlockPos pos)
    {
        buf.writeInt(pos.func_177958_n());
        buf.writeInt(pos.func_177956_o());
        buf.writeInt(pos.func_177952_p());
    }

    /**
     * Read chunk coordinates from a {@link ByteBuf}.
     *
     * @param buf Buf to read from.
     * @return Chunk coordinate that was read.
     */
    @NotNull
    public static BlockPos readFromByteBuf(@NotNull final ByteBuf buf)
    {
        final int x = buf.readInt();
        final int y = buf.readInt();
        final int z = buf.readInt();
        return new BlockPos(x, y, z);
    }

    /**
     * this checks that you are not in liquid.  Will check for all liquids, even those from other mods
     * before TP
     *
     * @param blockPos for the current block LOC
     * @param sender uses the player to get the world
     * @return isSafe true=safe false=water or lava
     */
    public static boolean isPositionSafe(@NotNull ICommandSender sender, BlockPos blockPos)
    {
        return sender.func_130014_f_().func_180495_p(blockPos).func_177230_c() != Blocks.field_150350_a
                && !sender.func_130014_f_().func_180495_p(blockPos).func_185904_a().func_76224_d()
                && !sender.func_130014_f_().func_180495_p(blockPos.func_177984_a()).func_185904_a().func_76224_d();
    }

    /**
     * this checks that you are not in the air or underground.
     * If so it will look up and down for a good landing spot before TP.
     *
     * @param blockPos for the current block LOC.
     * @param world the world to search in.
     * @return blockPos to be used for the TP.
     */
    public static BlockPos findLand(final BlockPos blockPos, final World world)
    {
        int top = blockPos.func_177956_o();
        int bot = 0;
        int mid = blockPos.func_177956_o();

        BlockPos foundland = null;
        BlockPos tempPos = blockPos;
        //We are doing a binary search to limit the amount of checks (usually at most 9 this way)
        while (top >= bot)
        {
            tempPos = new BlockPos( tempPos.func_177958_n(),mid, tempPos.func_177952_p());
            final Block blocks = world.func_180495_p(tempPos).func_177230_c();
            if (blocks == Blocks.field_150350_a && world.func_175678_i(tempPos))
            {
                top = mid - 1;
                foundland = tempPos;
            }
            else
            {
                bot = mid + 1;
                foundland = tempPos;
            }
            mid = (bot + top)/2;
        }

        return foundland;
    }

    /**
     * Returns if the {@link #getDistanceSquared(BlockPos, BlockPos)} from a coordinate to an citizen is closer than 4.84.
     *
     * @param coordinate Coordinate you want check distance of.
     * @param citizen    Citizen you want check distance of.
     * @return Whether or not the distance is less than 4.84.
     */
    public static boolean isClose(@NotNull final BlockPos coordinate, @NotNull final EntityCitizen citizen)
    {
        return getDistanceSquared(coordinate, citizen.func_180425_c()) < CLOSE_DISTANCE;
    }

    /**
     * Squared distance between two BlockPos.
     *
     * @param block1 position one.
     * @param block2 position two.
     * @return squared distance.
     */
    public static long getDistanceSquared(@NotNull final BlockPos block1, @NotNull final BlockPos block2)
    {
        final long xDiff = (long) block1.func_177958_n() - block2.func_177958_n();
        final long yDiff = (long) block1.func_177956_o() - block2.func_177956_o();
        final long zDiff = (long) block1.func_177952_p() - block2.func_177952_p();

        final long result = xDiff * xDiff + yDiff * yDiff + zDiff * zDiff;
        if (result < 0)
        {
            throw new IllegalStateException("max-sqrt is to high! Failure to catch overflow with "
                                              + xDiff + " | " + yDiff + " | " + zDiff);
        }
        return result;
    }

    /**
     * Simple two dimensional distance between two points.
     *
     * @param block1 position one.
     * @param block2 position two.
     * @return squared distance.
     */
    public static long getDistance(@NotNull BlockPos block1, @NotNull BlockPos block2)
    {
        final long xDiff = (long) block1.func_177958_n() - block2.func_177958_n();
        final long yDiff = (long) block1.func_177956_o() - block2.func_177956_o();
        final long zDiff = (long) block1.func_177952_p() - block2.func_177952_p();

        return Math.abs(xDiff + yDiff + zDiff);
    }

    /**
     * Squared distance between two BlockPos.
     *
     * @param block1 position one.
     * @param block2 position two.
     * @return squared distance.
     */
    public static long getDistance2D(@NotNull final BlockPos block1, @NotNull final BlockPos block2)
    {
        final long xDiff = Math.abs((long) block1.func_177958_n() - block2.func_177958_n());
        final long zDiff = Math.abs((long) block1.func_177952_p() - block2.func_177952_p());

        return Math.abs(xDiff + zDiff);
    }

    /**
     * 2D Squared distance between two BlockPos.
     *
     * @param block1 position one.
     * @param block2 position two.
     * @return 2D squared distance.
     */
    public static long getDistanceSquared2D(@NotNull final BlockPos block1, @NotNull final BlockPos block2)
    {
        final long xDiff = (long) block1.func_177958_n() - block2.func_177958_n();
        final long zDiff = (long) block1.func_177952_p() - block2.func_177952_p();

        final long result = xDiff * xDiff + zDiff * zDiff;
        if (result < 0)
        {
            throw new IllegalStateException("max-sqrt is to high! Failure to catch overflow with "
                                              + xDiff + " | " + zDiff);
        }
        return result;
    }

    /**
     * Returns the tile entity at a specific chunk coordinate.
     *
     * @param world World the tile entity is in.
     * @param pos   Coordinates of the tile entity.
     * @return Tile entity at the given coordinates.
     */
    public static TileEntity getTileEntity(@NotNull final World world, @NotNull final BlockPos pos)
    {
        return world.func_175625_s(pos);
    }

    /**
     * Returns a list of drops possible mining a specific block with specific fortune level.
     *
     * @param world   World the block is in.
     * @param coords  Coordinates of the block.
     * @param fortune Level of fortune on the pickaxe.
     * @return List of {@link ItemStack} with possible drops.
     */
    public static List<ItemStack> getBlockDrops(@NotNull final World world, @NotNull final BlockPos coords, final int fortune)
    {
        return getBlock(world, coords).getDrops(world, new BlockPos(coords.func_177958_n(), coords.func_177956_o(), coords.func_177952_p()), getBlockState(world, coords), fortune);
    }

    /**
     * Returns the block at a specific chunk coordinate.
     *
     * @param world  World the block is in.
     * @param coords Coordinates of the block.
     * @return Block at the given coordinates.
     */
    public static Block getBlock(@NotNull final World world, @NotNull final BlockPos coords)
    {
        return world.func_180495_p(coords).func_177230_c();
    }

    /**
     * Returns the metadata of a block at a specific chunk coordinate.
     *
     * @param world  World the block is in.
     * @param coords Coordinates of the block.
     * @return Metadata of the block at the given coordinates.
     */
    public static IBlockState getBlockState(@NotNull final World world, @NotNull final BlockPos coords)
    {
        return world.func_180495_p(coords);
    }

    /**
     * Sets a block in the world.
     *
     * @param world  World the block needs to be set in.
     * @param coords Coordinate to place block.
     * @param block  Block to place.
     * @return True if block is placed, otherwise false.
     */
    public static boolean setBlock(@NotNull final World world, final BlockPos coords, @NotNull final Block block)
    {
        return world.func_175656_a(coords, block.func_176223_P());
    }

    /**
     * Sets a block in the world, with specific metadata and flags.
     *
     * @param worldIn World the block needs to be set in.
     * @param coords  Coordinate to place block.
     * @param state   BlockState to be placed.
     * @param flag    Flag to set.
     * @return True if block is placed, otherwise false.
     */
    public static boolean setBlock(@NotNull final World worldIn, @NotNull final BlockPos coords, final IBlockState state, final int flag)
    {
        return worldIn.func_180501_a(coords, state, flag);
    }

    /**
     * Returns whether or not the citizen is heading to a specific location.
     * {@link EntityUtils#isPathingTo(EntityCitizen, int, int)}.
     *
     * @param citizen Citizen you want to check.
     * @param pos     Position you want to check.
     * @return True if citizen heads to pos, otherwise false.
     */
    public static boolean isPathingTo(@NotNull final EntityCitizen citizen, @NotNull final BlockPos pos)
    {
        return EntityUtils.isPathingTo(citizen, pos.func_177958_n(), pos.func_177952_p());
    }

    /**
     * {@link EntityUtils#isWorkerAtSiteWithMove(EntityCitizen, int, int, int)}.
     *
     * @param worker Worker to check.
     * @param site   Chunk coordinates of site to check.
     * @return True when worker is at site, otherwise false.
     */
    public static boolean isWorkerAtSiteWithMove(@NotNull final EntityCitizen worker, @NotNull final BlockPos site)
    {
        return EntityUtils.isWorkerAtSiteWithMove(worker, site.func_177958_n(), site.func_177956_o(), site.func_177952_p());
    }

    /**
     * {@link EntityUtils#isWorkerAtSiteWithMove(EntityCitizen, int, int, int, int)}.
     *
     * @param worker Worker to check.
     * @param site   Chunk coordinates of site to check.
     * @param range  Range to check in.
     * @return True when within range, otherwise false.
     */
    public static boolean isWorkerAtSiteWithMove(@NotNull final EntityCitizen worker, @NotNull final BlockPos site, final int range)
    {
        return EntityUtils.isWorkerAtSiteWithMove(worker, site.func_177958_n(), site.func_177956_o(), site.func_177952_p(), range);
    }

    /**
     * {@link EntityUtils#tryMoveLivingToXYZ(EntityLiving, int, int, int)}.
     *
     * @param living      A living entity.
     * @param destination chunk coordinates to check moving to.
     * @return True when XYZ is found, an set moving to, otherwise false.
     */
    public static boolean tryMoveLivingToXYZ(@NotNull final EntityLiving living, @NotNull final BlockPos destination)
    {
        return EntityUtils.tryMoveLivingToXYZ(living, destination.func_177958_n(), destination.func_177956_o(), destination.func_177952_p());
    }

    /**
     * Attempt to move to XYZ.
     * True when found and destination is set.
     *
     * @param citizen     Citizen to move to XYZ.
     * @param destination Chunk coordinate of the distance.
     * @return True when found, and destination is set, otherwise false.
     */
    public static PathResult moveLivingToXYZ(@NotNull final EntityCitizen citizen, @NotNull final BlockPos destination)
    {
        return citizen.func_70661_as().moveToXYZ(destination.func_177958_n(), destination.func_177956_o(), destination.func_177952_p(), 1.0);
    }

    /**
     * Create a method for using a {@link BlockPos} when using {@link net.minecraft.util.math.BlockPos.MutableBlockPos#setPos(int, int, int)}.
     *
     * @param pos    {@link net.minecraft.util.math.BlockPos.MutableBlockPos}.
     * @param newPos The new position to set.
     */
    public static void set(@NotNull final BlockPos.MutableBlockPos pos, @NotNull final BlockPos newPos)
    {
        pos.func_181079_c(newPos.func_177958_n(), newPos.func_177956_o(), newPos.func_177952_p());
    }

    /**
     * Returns whether a chunk coordinate is equals to (x, y, z).
     *
     * @param coords Chunk Coordinate    (point 1).
     * @param x      x-coordinate        (point 2).
     * @param y      y-coordinate        (point 2).
     * @param z      z-coordinate        (point 2).
     * @return True when coordinates are equal, otherwise false.
     */
    public static boolean isEqual(@NotNull final BlockPos coords, final int x, final int y, final int z)
    {
        return coords.func_177958_n() == x && coords.func_177956_o() == y && coords.func_177952_p() == z;
    }

    /**
     * Returns the Chunk Coordinate created from an entity.
     *
     * @param entity Entity to create chunk coordinates from.
     * @return Chunk Coordinates created from the entity.
     */
    @NotNull
    public static BlockPos fromEntity(@NotNull final Entity entity)
    {
        return new BlockPos(MathHelper.func_76128_c(entity.field_70165_t), MathHelper.func_76128_c(entity.field_70163_u), MathHelper.func_76128_c(entity.field_70161_v));
    }
}
