package com.minecolonies.coremod.util;

import com.minecolonies.coremod.blocks.AbstractBlockHut;
import net.minecraft.block.*;
import net.minecraft.block.properties.PropertyBool;
import net.minecraft.block.state.IBlockState;
import net.minecraft.init.Blocks;
import net.minecraft.init.Items;
import net.minecraft.item.*;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.Rotation;
import net.minecraft.util.math.BlockPos;
import net.minecraft.world.World;
import net.minecraftforge.fml.common.registry.GameData;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.function.BiPredicate;

/**
 * Utility class for all Block type checking.
 */
public final class BlockUtils
{
    /**
     * Predicated to determine if a block is free to place.
     */
    @NotNull
    private static final List<BiPredicate<Block, IBlockState>> freeToPlaceBlocks =
            Arrays.asList(
                    (block, iBlockState) -> block.equals(Blocks.field_150350_a),
                    (block, iBlockState) -> iBlockState.func_185904_a().func_76224_d(),
                    (block, iBlockState) -> BlockUtils.isWater(block.func_176223_P()),
                    (block, iBlockState) -> block.equals(Blocks.field_150362_t),
                    (block, iBlockState) -> block.equals(Blocks.field_150361_u),
                    (block, iBlockState) -> block.equals(Blocks.field_150398_cm),
                    (block, iBlockState) -> block.equals(Blocks.field_150349_c),
                    (block, iBlockState) -> block instanceof BlockDoor
                            && iBlockState != null
                            && iBlockState.func_177229_b(PropertyBool.func_177716_a("upper"))

            );

    /**
     * Private constructor to hide the public one.
     */
    private BlockUtils()
    {
        //Hides implicit constructor.
    }

    /**
     * Updates the rotation of the structure depending on the input.
     *
     * @param rotation the rotation to be set.
     * @return returns the Rotation object.
     */
    public static Rotation getRotation(final int rotation)
    {
        switch (rotation)
        {
            case 1:
                return Rotation.CLOCKWISE_90;
            case 2:
                return Rotation.CLOCKWISE_180;
            case 3:
                return Rotation.COUNTERCLOCKWISE_90;
            default:
                return Rotation.NONE;
        }
    }

    /**
     * Get the filler block at a certain location.
     * If block follows gravity laws return dirt.
     * @param world the world the block is in.
     * @param location the location it is at.
     * @return the IBlockState of the filler block.
     */
    public static IBlockState getSubstitutionBlockAtWorld(@NotNull final World world, @NotNull final BlockPos location)
    {
        final IBlockState filler = world.func_180494_b(location).field_76753_B;
        if (filler.func_177230_c() instanceof BlockFalling)
        {
            return Blocks.field_150346_d.func_176223_P();
        }
        return filler;
    }

    /**
     * Checks if this block type should be destroyed.
     * <p>
     * The builder uses this to check if he should clear this block.
     *
     * @param block the block type to check
     * @return true if you should back away
     */
    public static boolean shouldNeverBeMessedWith(final Block block)
    {
        return block instanceof AbstractBlockHut
                || Objects.equals(block, Blocks.field_150357_h);
    }

    /**
     * Gets a rotation from a block facing.
     *
     * @param facing the block facing.
     * @return the int rotation.
     */
    public static int getRotationFromFacing(final EnumFacing facing)
    {
        switch (facing)
        {
            case SOUTH:
                return 2;
            case EAST:
                return 1;
            case WEST:
                return 3;
            default:
                return 0;
        }
    }

    /**
     * Checks if this block type is something we can place for free.
     * <p>
     * The builder uses this to determine if he need resources for the block.
     *
     * @param block the block to check.
     * @return true if we can just place it.
     */
    public static boolean freeToPlace(final Block block)
    {
        return freeToPlace(block, null);
    }

    /**
     * Checks if this block type is something we can place for free.
     * <p>
     * The builder uses this to determine if he need resources for the block.
     *
     * @param block      the block to check.
     * @param blockState the state this block has.
     * @return true if we can just place it.
     */
    public static boolean freeToPlace(@Nullable final Block block, final IBlockState blockState)
    {
        if (block == null)
        {
            return true;
        }
        for (@NotNull final BiPredicate<Block, IBlockState> predicate : freeToPlaceBlocks)
        {
            if (predicate.test(block, blockState))
            {
                return true;
            }
        }
        return false;
    }

    /**
     * Checks if the block is water.
     *
     * @param iBlockState block state to be checked.
     * @return true if is water.
     */
    public static boolean isWater(final IBlockState iBlockState)
    {
        return Objects.equals(iBlockState, Blocks.field_150355_j.func_176223_P())
                || Objects.equals(iBlockState, Blocks.field_150358_i.func_176223_P());
    }

    /**
     * Checks if a certain block returns a seed as the item.
     *
     * @param world the world the block is in.
     * @param pos   the position the block is at.
     * @return true if is a seed.
     */
    public static boolean isBlockSeed(@NotNull final World world, @NotNull final BlockPos pos)
    {
        return BlockUtils.getItemStackFromBlockState(world.func_180495_p(pos.func_177984_a())) != null
                && BlockUtils.getItemStackFromBlockState(world.func_180495_p(pos.func_177984_a())).func_77973_b() instanceof ItemSeeds;
    }

    /**
     * Mimics pick block.
     *
     * @param blockState the block and state we are creating an ItemStack for.
     * @return ItemStack fromt the BlockState.
     */
    public static ItemStack getItemStackFromBlockState(@NotNull final IBlockState blockState)
    {
        final Item item = getItem(blockState);

        if (item == null)
        {
            return null;
        }

        Block block = blockState.func_177230_c();
        if (item instanceof ItemBlock)
        {
            block = Block.func_149634_a(item);
        }

        return new ItemStack(item, 1, getDamageValue(block, blockState));
    }

    private static Item getItem(@NotNull final IBlockState blockState)
    {
        if (blockState.func_177230_c() instanceof BlockBanner)
        {
            return Items.field_179564_cE;
        }
        else if (blockState.func_177230_c() instanceof BlockBed)
        {
            return Items.field_151104_aV;
        }
        else if (blockState.func_177230_c() instanceof BlockBrewingStand)
        {
            return Items.field_151067_bt;
        }
        else if (blockState.func_177230_c() instanceof BlockCake)
        {
            return Items.field_151105_aU;
        }
        else if (blockState.func_177230_c() instanceof BlockCauldron)
        {
            return Items.field_151066_bu;
        }
        else if (blockState.func_177230_c() instanceof BlockCocoa)
        {
            return Items.field_151100_aR;
        }
        else if (blockState.func_177230_c() instanceof BlockCrops)
        {
            final ItemStack stack = ((BlockCrops) blockState.func_177230_c()).func_185473_a(null, null, blockState);
            if (stack != null)
            {
                return stack.func_77973_b();
            }

            return Items.field_151014_N;
        }
        else if (blockState.func_177230_c() instanceof BlockDaylightDetector)
        {
            return Item.func_150898_a(Blocks.field_150453_bW);
        }
        else if (blockState.func_177230_c() instanceof BlockDoor)
        {
            final Item item = blockState.func_177230_c() == Blocks.field_150454_av ? Items.field_151139_aw
                    : (blockState.func_177230_c() == Blocks.field_180414_ap ? Items.field_179569_ar
                            : (blockState.func_177230_c() == Blocks.field_180412_aq ? Items.field_179568_as
                                    : (blockState.func_177230_c() == Blocks.field_180411_ar ? Items.field_179567_at
                                            : (blockState.func_177230_c() == Blocks.field_180410_as ? Items.field_179572_au
                                                    : (blockState.func_177230_c() == Blocks.field_180409_at
                                                            ? Items.field_179571_av
                                                            : Items.field_179570_aq)))));

            return item == null ? Item.func_150898_a(blockState.func_177230_c()) : item;
        }
        else if (blockState.func_177230_c() instanceof BlockFarmland || blockState.func_177230_c() instanceof BlockGrassPath)
        {
            return Item.func_150898_a(Blocks.field_150346_d);
        }
        else if (blockState.func_177230_c() instanceof BlockFlowerPot)
        {
             return Items.field_151162_bE;
        }
        else if (blockState.func_177230_c() instanceof BlockFurnace)
        {
            return Item.func_150898_a(Blocks.field_150460_al);
        }
        else if (blockState.func_177230_c() instanceof BlockHugeMushroom)
        {
            // Can the builder even build this?
            return blockState.func_177230_c().func_180660_a(null, null, 0);
        }
        else if (blockState.func_177230_c() instanceof BlockNetherWart)
        {
            return Items.field_151075_bm;
        }
        else if (blockState.func_177230_c() instanceof BlockPistonExtension)
        {
            // Not really sure what we want to do here...
            return blockState.func_177229_b(BlockPistonExtension.field_176325_b) == BlockPistonExtension.EnumPistonType.STICKY
                    ? Item.func_150898_a(Blocks.field_150320_F)
                    : Item.func_150898_a(Blocks.field_150331_J);
        }
        else if (blockState.func_177230_c() instanceof BlockRedstoneComparator)
        {
            return Items.field_151132_bS;
        }
        else if (blockState.func_177230_c() instanceof BlockRedstoneLight)
        {
            return Item.func_150898_a(Blocks.field_150379_bu);
        }
        else if (blockState.func_177230_c() instanceof BlockRedstoneRepeater)
        {
            return Items.field_151107_aW;
        }
        else if (blockState.func_177230_c() instanceof BlockRedstoneTorch)
        {
            return Item.func_150898_a(Blocks.field_150429_aA);
        }
        else if (blockState.func_177230_c() instanceof BlockRedstoneWire)
        {
            return Items.field_151137_ax;
        }
        else if (blockState.func_177230_c() instanceof BlockReed)
        {
            return Items.field_151120_aE;
        }
        else if (blockState.func_177230_c() instanceof BlockSign)
        {
            return Items.field_151155_ap;
        }
        else if (blockState.func_177230_c() instanceof BlockSkull)
        {
            return Items.field_151144_bL;
        }
        else if (blockState.func_177230_c() instanceof BlockStem)
        {
            final ItemStack stack = ((BlockStem) blockState.func_177230_c()).func_185473_a(null, null, blockState);
            if (stack != null)
            {
                return stack.func_77973_b();
            }
            return Items.field_151081_bc;
        }
        else if (blockState.func_177230_c() instanceof BlockStoneSlab)
        {
            //Builder won't know how to build double stone slab
            return Item.func_150898_a(Blocks.field_150333_U);
        }
        else if (blockState.func_177230_c() instanceof BlockPurpurSlab)
        {
            return Item.func_150898_a(Blocks.field_185771_cX);
        }
        else if (blockState.func_177230_c() instanceof BlockStoneSlabNew)
        {
            return Item.func_150898_a(Blocks.field_180389_cP);
        }
        else if (blockState.func_177230_c() instanceof BlockTripWire)
        {
            return Items.field_151007_F;
        }
        else if (blockState.func_177230_c() instanceof BlockWoodSlab)
        {
            //Builder will also have trouble with double wood slab
            return Item.func_150898_a(Blocks.field_150376_bx);
        }
        else
        {
            return GameData.getBlockItemMap().get(blockState.func_177230_c());
        }
    }

    /**
     * Get the damage value from a block and blockState, where the block is the placeable and obtainable block.
     * The blockstate might differ from the block.
     * @param block the block.
     * @param blockState the state.
     * @return the int damage value.
     */
    private static int getDamageValue(final Block block, @NotNull final IBlockState blockState)
    {
        if (block instanceof BlockCocoa)
        {
            return EnumDyeColor.BROWN.func_176767_b();
        }
        else if (block instanceof BlockDirt && !(blockState.func_177230_c() instanceof BlockFarmland))
        {
            return blockState.func_177229_b(BlockDirt.field_176386_a).func_176925_a();
        }
        else if (block instanceof BlockDoublePlant
                && blockState.func_177229_b(BlockDoublePlant.field_176492_b) == BlockDoublePlant.EnumBlockHalf.LOWER)
        {
            //If upper part we can't do much here
            return blockState.func_177229_b(BlockDoublePlant.field_176493_a).func_176936_a();
        }
        else if (block instanceof BlockNewLeaf)
        {
            return block.func_176201_c(blockState) & 3;
        }
        else if (block instanceof BlockOre)
        {
            return 0;
        }
        else if (block instanceof BlockSilverfish || block instanceof BlockTallGrass)
        {
            return block.func_176201_c(blockState);
        }
        else if (block instanceof BlockSlab)
        {
            return block.func_180651_a(blockState) & 7;
        }
        else
        {
            return block.func_180651_a(blockState);
        }
    }

    /**
     * Checks if a certain block is a pathBlock (roadBlock).
     *
     * @param block the block to analyze.
     * @return true if is so.
     */
    public static boolean isPathBlock(final Block block)
    {
        return block == Blocks.field_150351_n || block == Blocks.field_150417_aV;
    }
}
