package com.minecolonies.coremod.util;

import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.util.text.ITextComponent;
import net.minecraft.util.text.TextComponentTranslation;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.List;

/**
 * Helper class for localization and sending player messages.
 *
 * @author Colton
 */
public final class LanguageHandler
{
    /**
     * Private constructor to hide implicit one.
     */
    private LanguageHandler()
    {
        // Intentionally left empty.
    }

    /**
     * Send a message to the player.
     *
     * @param player the player to send to.
     * @param key the key of the message.
     * @param message the message to send.
     */
    public static void sendPlayerMessage(@NotNull final EntityPlayer player, final String key, final Object... message)
    {
        player.func_146105_b(buildChatComponent(key, message));
    }

    /**
     * Localize a string and use String.format().
     *
     * @param key  translation key.
     * @param args Objects for String.format().
     * @return Localized string.
     */
    public static String format(final String key, final Object... args)
    {
        final String result;
        if(args.length == 0)
        {
            result = new TextComponentTranslation(key).func_150260_c();
        }
        else
        {
            result = new TextComponentTranslation(key, args).func_150260_c();
        }
        return result.isEmpty() ? key : result;
    }

    /**
     * Send message to a list of players.
     *
     * @param players the list of players.
     * @param key key of the message.
     * @param message the message.
     */
    public static void sendPlayersMessage(@Nullable final List<EntityPlayer> players, final String key, final Object... message)
    {
        if (players == null || players.isEmpty())
        {
            return;
        }

        ITextComponent textComponent = buildChatComponent(key, message);

        for (@NotNull final EntityPlayer player : players)
        {
            player.func_146105_b(textComponent);
        }
    }

    private static ITextComponent buildChatComponent(final String key, final Object... message)
    {
        TextComponentTranslation translation = null;

        int onlyArgsUntil = 0;
        for (final Object object : message)
        {
            if (object instanceof ITextComponent)
            {
                if (onlyArgsUntil == 0)
                {
                    onlyArgsUntil = -1;
                }
                break;
            }
            onlyArgsUntil++;
        }

        if (onlyArgsUntil >= 0)
        {
            final Object[] args = new Object[onlyArgsUntil];
            System.arraycopy(message, 0, args, 0, onlyArgsUntil);

            translation = new TextComponentTranslation(key, args);
        }

        for (final Object object : message)
        {
            if (translation == null)
            {
                if (object instanceof ITextComponent)
                {
                    translation = new TextComponentTranslation(key);
                }
                else
                {
                    translation = new TextComponentTranslation(key, object);
                    continue;
                }
            }

            if (object instanceof ITextComponent)
            {
                translation.func_150257_a((ITextComponent) object);
            }
            else if (object instanceof String)
            {
                boolean isInArgs = false;
                for (final Object obj : translation.func_150271_j())
                {
                    if (obj.equals(object))
                    {
                        isInArgs = true;
                        break;
                    }
                }

                if(!isInArgs)
                {
                    translation.func_150258_a((String) object);
                }
            }
        }

        if (translation == null)
        {
            translation = new TextComponentTranslation(key);
        }

        return translation;
    }
}
