package com.minecolonies.structures.helpers;

import com.minecolonies.coremod.blocks.AbstractBlockHut;
import com.minecolonies.coremod.util.BlockPosUtil;
import net.minecraft.block.state.IBlockState;
import net.minecraft.util.Mirror;
import net.minecraft.util.Rotation;
import net.minecraft.util.math.BlockPos;
import net.minecraft.world.World;
import net.minecraft.world.gen.structure.template.PlacementSettings;
import net.minecraft.world.gen.structure.template.Template;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.List;

/**
 * Proxy class translating the structures method to something we can use.
 */
public class StructureProxy
{
    private final Structure structure;
    private Template.EntityInfo[][][] entities;
    private Template.BlockInfo[][][] blocks;
    private int                      width;
    private int                      height;
    private int                      length;
    private BlockPos                 offset;

    /**
     * @param worldObj the world.
     * @param name     the string where the structure is saved at.
     */
    public StructureProxy(final World worldObj, final String name)
    {
        this.structure = new Structure(worldObj, name, new PlacementSettings());

        if(structure.isTemplateMissing())
        {
            return;
        }
        final BlockPos size = structure.getSize(Rotation.NONE);

        this.width = size.func_177958_n();
        this.height = size.func_177956_o();
        this.length = size.func_177952_p();

        this.blocks = new Template.BlockInfo[width][height][length];
        this.entities = new Template.EntityInfo[width][height][length];

        for (final Template.BlockInfo info : structure.getBlockInfo())
        {
            final BlockPos tempPos = info.field_186242_a;
            blocks[tempPos.func_177958_n()][tempPos.func_177956_o()][tempPos.func_177952_p()] = info;
            entities[tempPos.func_177958_n()][tempPos.func_177956_o()][tempPos.func_177952_p()] = null;

            if (info.field_186243_b.func_177230_c() instanceof AbstractBlockHut)
            {
                offset = info.field_186242_a;
            }
        }

        for(final Template.EntityInfo info: structure.getTileEntities())
        {
            final BlockPos tempPos = info.field_186248_b;
            entities[tempPos.func_177958_n()][tempPos.func_177956_o()][tempPos.func_177952_p()] = info;
        }
    }

    /**
     * Getter of the offset.
     *
     * @return the blockPos of the offset.
     */
    public BlockPos getOffset()
    {
        return offset;
    }

    /**
     * Setter of the offset.
     *
     * @param pos the new offset.
     */
    public void setOffset(final BlockPos pos)
    {
        offset = pos;
    }

    /**
     * Getter of the type of the structure.
     *
     * @return true if so.
     */
    public String getType()
    {
        if (hasOffset())
        {
            return "Hut";
        }
        return "Decoration";
    }

    /**
     * Checks if the structure has an offset.
     *
     * @return true if so.
     */
    private boolean hasOffset()
    {
        return !BlockPosUtil.isEqual(offset, 0, 0, 0);
    }

    /**
     * Getter for the structure.
     * @return the structure object.
     */
    public Structure getStructure()
    {
        return structure;
    }

    /**
     * Getter of the IBlockState at a certain position.
     *
     * @param pos the position.
     * @return the blockState.
     */
    public IBlockState getBlockState(@NotNull final BlockPos pos)
    {
        return blocks[pos.func_177958_n()][pos.func_177956_o()][pos.func_177952_p()].field_186243_b;
    }

    /**
     * Getter of the BlockInfo at a certain position.
     *
     * @param pos the position.
     * @return the blockState.
     */
    public Template.BlockInfo getBlockInfo(@NotNull final BlockPos pos)
    {
        return blocks[pos.func_177958_n()][pos.func_177956_o()][pos.func_177952_p()];
    }

    /**
     * Return a list of tileEntities.
     *
     * @return list of them.
     */
    public List<Template.EntityInfo> getTileEntities()
    {
        return this.structure.getTileEntities();
    }

    /**
     * Getter of the EntityInfo at a certain position.
     *
     * @param pos the position.
     * @return the blockState.
     */
    @Nullable
    public Template.EntityInfo getEntityinfo(@NotNull final BlockPos pos)
    {
        if(entities[pos.func_177958_n()][pos.func_177956_o()].length == 0)
        {
            return null;
        }
        return entities[pos.func_177958_n()][pos.func_177956_o()][pos.func_177952_p()];
    }

    /**
     * Getter of the width.
     *
     * @return the width.
     */
    public int getWidth()
    {
        return this.width;
    }

    /**
     * Getter of the length.
     *
     * @return the length
     */
    public int getLength()
    {
        return this.length;
    }

    /**
     * Getter of the height.
     *
     * @return the height
     */
    public int getHeight()
    {
        return this.height;
    }

    /**
     * Rotate the structure depending on the direction it's facing.
     *
     * @param times times to rotateWithMirror.
     * @param world the world to rotateWithMirror it in.
     * @param rotatePos the pos to rotateWithMirror it around.
     * @param mirror the mirror
     */
    public void rotateWithMirror(final int times, World world, BlockPos rotatePos, Mirror mirror)
    {
        final Rotation rotation;
        switch (times)
        {
            case 1:
                rotation = Rotation.CLOCKWISE_90;
                break;
            case 2:
                rotation = Rotation.CLOCKWISE_180;
                break;
            case 3:
                rotation = Rotation.COUNTERCLOCKWISE_90;
                break;
            default:
                rotation = Rotation.NONE;
        }
        structure.setPlacementSettings(new PlacementSettings().func_186220_a(rotation).func_186214_a(mirror));

        final BlockPos size = structure.getSize(rotation);

        this.width = size.func_177958_n();
        this.height = size.func_177956_o();
        this.length = size.func_177952_p();

        this.blocks = new Template.BlockInfo[width][height][length];
        this.entities = new Template.EntityInfo[width][height][length];

        int minX = 0;
        int minY = 0;
        int minZ = 0;

        for (final Template.BlockInfo info : structure.getBlockInfoWithSettings(new PlacementSettings().func_186220_a(rotation).func_186214_a(mirror)))
        {
            final BlockPos tempPos = info.field_186242_a;
            final int x = tempPos.func_177958_n();
            final int y = tempPos.func_177956_o();
            final int z = tempPos.func_177952_p();
            if (x < minX)
            {
                minX = x;
            }

            if (y < minY)
            {
                minY = y;
            }

            if (z < minZ)
            {
                minZ = z;
            }
        }

        minX = Math.abs(minX);
        minY = Math.abs(minY);
        minZ = Math.abs(minZ);
        boolean foundHut = false;
        final PlacementSettings settings = new PlacementSettings().func_186220_a(rotation).func_186214_a(mirror);

        for (final Template.BlockInfo info : structure.getBlockInfoWithSettings(settings))
        {
            final BlockPos tempPos = info.field_186242_a;
            final int x = tempPos.func_177958_n() + minX;
            final int y = tempPos.func_177956_o() + minY;
            final int z = tempPos.func_177952_p() + minZ;

            this.blocks[x][y][z] = info;
            this.entities[x][y][z] = null;

            if (info.field_186243_b.func_177230_c() instanceof AbstractBlockHut)
            {
                foundHut = true;
                offset = info.field_186242_a.func_177982_a(minX, minY, minZ);
            }
        }

        updateOffSetIfDecoration(foundHut, size, times, minX, minY, minZ);

        for(final Template.EntityInfo info: structure.getTileEntities())
        {
            final Template.EntityInfo newInfo = structure.transformEntityInfoWithSettings(info, world, rotatePos.func_177973_b(offset).func_177971_a(new BlockPos(minX, minY, minZ)), settings);
            //289 74 157 - 289.9 76.5, 157.5
            final BlockPos tempPos = Template.func_186266_a(settings, info.field_186248_b);
            final int x = tempPos.func_177958_n() + minX;
            final int y = tempPos.func_177956_o() + minY;
            final int z = tempPos.func_177952_p() + minZ;
            this.entities[x][y][z] = newInfo;
        }
    }

    /**
     * Updates the offset if the structure is a decoration.
     *
     * @param foundHut if false update.
     */
    private void updateOffSetIfDecoration(final boolean foundHut, final BlockPos size, int rotation, int minX, int minY, int minZ)
    {
        if (!foundHut)
        {
            BlockPos tempSize = size;
            if (rotation == 1)
            {
                tempSize = new BlockPos(-size.func_177958_n(), size.func_177956_o(), size.func_177952_p());
            }
            if (rotation == 2)
            {
                tempSize = new BlockPos(-size.func_177958_n(), size.func_177956_o(), -size.func_177952_p());
            }
            if (rotation == 3)
            {
                tempSize = new BlockPos(size.func_177958_n(), size.func_177956_o(), -size.func_177952_p());
            }

            offset = new BlockPos(tempSize.func_177958_n() / 2, 0, tempSize.func_177952_p() / 2).func_177982_a(minX, minY, minZ);
        }
    }
}
