package com.minecolonies.blockout.controls;

import com.minecolonies.blockout.Pane;
import com.minecolonies.blockout.PaneParams;
import com.minecolonies.blockout.View;
import net.minecraft.client.gui.GuiScreen;
import net.minecraft.client.renderer.GlStateManager;
import net.minecraft.client.renderer.Tessellator;
import net.minecraft.client.renderer.VertexBuffer;
import net.minecraft.client.renderer.vertex.DefaultVertexFormats;
import net.minecraft.util.ResourceLocation;
import net.minecraft.util.math.MathHelper;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.lwjgl.input.Keyboard;
import org.lwjgl.opengl.GL11;

/**
 * Class which can be used to add text fields to a pane.
 */
public class TextField extends Pane
{
    /**
     * Texture resource location.
     */
    private static final ResourceLocation TEXTURE           = new ResourceLocation("textures/gui/widgets.png");
    //  Attributes
    protected            int              maxTextLength     = 32;
    protected            int              textColor         = 0xE0E0E0;
    protected            int              textColorDisabled = 0x707070;
    protected            boolean          shadow            = true;
    @Nullable
    protected            String           tabNextPaneID     = null;
    //  Runtime
    protected            String           text              = "";
    protected Filter filter;
    protected int cursorPosition     = 0;
    protected int scrollOffset       = 0;
    protected int selectionEnd       = 0;
    protected int cursorBlinkCounter = 0;

    /**
     * Simple public constructor to instantiate.
     */
    public TextField()
    {
        super();
        //Required
    }

    /**
     * Public constructor to instantiate the field with params.
     * @param params the parameters for the textField.
     */
    public TextField(@NotNull final PaneParams params)
    {
        super(params);
        maxTextLength = params.getIntegerAttribute("maxlength", maxTextLength);
        textColor = params.getColorAttribute("color", textColor);
        textColorDisabled = params.getColorAttribute("colordisabled", textColorDisabled);
        shadow = params.getBooleanAttribute("shadow", shadow);
        text = params.getLocalizedStringAttribute("textContent", text);
        tabNextPaneID = params.getStringAttribute("tab", null);
    }

    public Filter getFilter()
    {
        return filter;
    }

    public void setFilter(final Filter f)
    {
        filter = f;
    }

    public String getText()
    {
        return text;
    }

    public void setText(@NotNull final String s)
    {
        text = s.length() <= maxTextLength ? s : s.substring(0, maxTextLength);
        setCursorPosition(text.length());
    }

    public int getInternalWidth()
    {
        return getWidth();
    }

    public int getMaxTextLength()
    {
        return maxTextLength;
    }

    public void setMaxTextLength(final int m)
    {
        maxTextLength = m;
    }

    public int getTextColor()
    {
        return textColor;
    }

    public void setTextColor(final int c)
    {
        textColor = c;
    }

    public int getTextColorDisabled()
    {
        return textColorDisabled;
    }

    public void setTextColorDisabled(final int c)
    {
        textColorDisabled = c;
    }

    @Nullable
    public String getTabNextPaneID()
    {
        return tabNextPaneID;
    }

    public void setTabNextPaneID(final String nextID)
    {
        tabNextPaneID = nextID;
    }

    public int getCursorPosition()
    {
        return cursorPosition;
    }

    public void setCursorPosition(final int pos)
    {
        cursorPosition = MathHelper.func_76125_a(pos, 0, text.length());
        setSelectionEnd(cursorPosition);
    }

    /**
     * Move the cursor by an offset.
     *
     * @param offset the offset.
     */
    public void moveCursorBy(final int offset)
    {
        setCursorPosition(selectionEnd + offset);
    }

    public int getSelectionEnd()
    {
        return selectionEnd;
    }

    public void setSelectionEnd(final int pos)
    {
        selectionEnd = MathHelper.func_76125_a(pos, 0, text.length());

        final int internalWidth = getInternalWidth();
        if (internalWidth > 0)
        {
            if (scrollOffset > text.length())
            {
                scrollOffset = text.length();
            }

            final String visibleString = mc.field_71466_p.func_78269_a(text.substring(scrollOffset), internalWidth);
            final int rightmostVisibleChar = visibleString.length() + scrollOffset;

            if (selectionEnd == scrollOffset)
            {
                scrollOffset -= mc.field_71466_p.func_78262_a(text, internalWidth, true).length();
            }

            if (selectionEnd > rightmostVisibleChar)
            {
                scrollOffset += selectionEnd - rightmostVisibleChar;
            }
            else if (selectionEnd <= scrollOffset)
            {
                scrollOffset -= scrollOffset - selectionEnd;
            }

            scrollOffset = MathHelper.func_76125_a(scrollOffset, 0, text.length());
        }
    }

    @NotNull
    public String getSelectedText()
    {
        final int start = Math.min(cursorPosition, selectionEnd);
        final int end = Math.max(cursorPosition, selectionEnd);
        return text.substring(start, end);
    }

    /**
     * Handle key event.
     *
     * @param c   the character.
     * @param key the key.
     * @return if it should be processed or not.
     */
    private boolean handleKey(final char c, final int key)
    {
        switch (key)
        {
            case Keyboard.KEY_BACK:
            case Keyboard.KEY_DELETE:
                return handleDelete(key);

            case Keyboard.KEY_HOME:
            case Keyboard.KEY_END:
                return handleHomeEnd(key);

            case Keyboard.KEY_LEFT:
            case Keyboard.KEY_RIGHT:
                return handleArrowKeys(key);

            case Keyboard.KEY_TAB:
                return handleTab();

            default:
                return handleChar(c);
        }
    }

    private boolean handleChar(final char c)
    {
        if (filter.isAllowedCharacter(c))
        {
            writeText(Character.toString(c));
            return true;
        }
        return false;
    }

    private boolean handleTab()
    {
        if (tabNextPaneID != null)
        {
            final Pane next = getWindow().findPaneByID(tabNextPaneID);
            if (next != null)
            {
                next.setFocus();
            }
        }
        return true;
    }

    private boolean handleArrowKeys(final int key)
    {
        final int direction = (key == Keyboard.KEY_LEFT) ? -1 : 1;

        if (GuiScreen.func_146272_n())
        {
            if (GuiScreen.func_146271_m())
            {
                setSelectionEnd(getNthWordFromPos(direction, getSelectionEnd()));
            }
            else
            {
                setSelectionEnd(getSelectionEnd() + direction);
            }
        }
        else if (GuiScreen.func_146271_m())
        {
            setCursorPosition(getNthWordFromCursor(direction));
        }
        else
        {
            moveCursorBy(direction);
        }
        return true;
    }

    private boolean handleHomeEnd(final int key)
    {
        final int position = (key == Keyboard.KEY_HOME) ? 0 : text.length();

        if (GuiScreen.func_146272_n())
        {
            setSelectionEnd(position);
        }
        else
        {
            setCursorPosition(position);
        }
        return true;
    }

    private boolean handleDelete(final int key)
    {
        final int direction = (key == Keyboard.KEY_BACK) ? -1 : 1;

        if (GuiScreen.func_146271_m())
        {
            deleteWords(direction);
        }
        else
        {
            deleteFromCursor(direction);
        }

        return true;
    }

    @Override
    public void onFocus()
    {
        setCursorPosition(text.length());
        cursorBlinkCounter = 0;
    }

    @Override
    /**
     * Draw itself at positions mx and my.
     */
    protected void drawSelf(final int mx, final int my)
    {
        final int color = enabled ? textColor : textColorDisabled;
        final int drawWidth = getInternalWidth();
        final int drawX = x;
        final int drawY = y;

        //  Determine the portion of the string that is visible on screen
        final String visibleString = mc.field_71466_p.func_78269_a(text.substring(scrollOffset), drawWidth);

        final int relativeCursorPosition = cursorPosition - scrollOffset;
        int relativeSelectionEnd = selectionEnd - scrollOffset;
        final boolean cursorVisible = relativeCursorPosition >= 0 && relativeCursorPosition <= visibleString.length();
        final boolean cursorBeforeEnd = cursorPosition < text.length() || text.length() >= maxTextLength;

        //  Enforce selection to the length limit of the visible string
        if (relativeSelectionEnd > visibleString.length())
        {
            relativeSelectionEnd = visibleString.length();
        }

        //  Draw string up through cursor
        int textX = drawX;
        if (visibleString.length() > 0)
        {
            @NotNull final String s1 = cursorVisible ? visibleString.substring(0, relativeCursorPosition) : visibleString;
            mc.field_71446_o.func_110577_a(TEXTURE);
            textX = mc.field_71466_p.func_175065_a(s1, textX, drawY, color, shadow);
        }

        int cursorX = textX;
        if (!cursorVisible)
        {
            cursorX = relativeCursorPosition > 0 ? (drawX + width) : drawX;
        }
        else if (cursorBeforeEnd)
        {
            if (shadow)
            {
                textX -= 1;
            }
            cursorX = textX;
        }

        //  Draw string after cursor
        if (visibleString.length() > 0 && cursorVisible && relativeCursorPosition < visibleString.length())
        {
            mc.field_71446_o.func_110577_a(TEXTURE);
            mc.field_71466_p.func_175065_a(visibleString.substring(relativeCursorPosition), textX, drawY, color, shadow);
        }

        //  Should we draw the cursor this frame?
        if (isFocus() && cursorVisible && (cursorBlinkCounter / 6 % 2 == 0))
        {
            if (cursorBeforeEnd)
            {
                func_73734_a(cursorX, drawY - 1, cursorX + 1, drawY + 1 + mc.field_71466_p.field_78288_b, -3092272);
            }
            else
            {
                mc.field_71446_o.func_110577_a(TEXTURE);
                mc.field_71466_p.func_175065_a("_", cursorX, drawY, color, shadow);
            }
        }

        //  Draw selection
        if (relativeSelectionEnd != relativeCursorPosition)
        {
            final int selectedDrawX = drawX + mc.field_71466_p.func_78256_a(visibleString.substring(0, relativeSelectionEnd));

            int selectionStartX = Math.min(cursorX, selectedDrawX - 1);
            int selectionEndX = Math.max(cursorX, selectedDrawX - 1);

            if (selectionStartX > (x + width))
            {
                selectionStartX = x + width;
            }

            if (selectionEndX > (x + width))
            {
                selectionEndX = x + width;
            }

            final Tessellator tessellator = Tessellator.func_178181_a();
            GlStateManager.func_179131_c(0.0F, 0.0F, 255.0F, 255.0F);
            GlStateManager.func_179090_x();
            GlStateManager.func_179115_u();
            GlStateManager.func_179116_f(GL11.GL_OR_REVERSE);
            final VertexBuffer vertexBuffer = tessellator.func_178180_c();

            // There are several to choose from, look at DefaultVertexFormats for more info
            vertexBuffer.func_181668_a(GL11.GL_QUADS, DefaultVertexFormats.field_181707_g);

            //Since our points do not have any u,v this seems to be the correct code
            vertexBuffer.func_181662_b((double) selectionStartX, (double) drawY + 1 + mc.field_71466_p.field_78288_b, 0.0D).func_181675_d();
            vertexBuffer.func_181662_b((double) selectionEndX, (double) drawY + 1 + mc.field_71466_p.field_78288_b, 0.0D).func_181675_d();
            vertexBuffer.func_181662_b((double) selectionEndX, (double) drawY - 1, 0.0D).func_181675_d();
            vertexBuffer.func_181662_b((double) selectionStartX, (double) drawY - 1, 0.0D).func_181675_d();
            tessellator.func_78381_a();
            GlStateManager.func_179134_v();
            GlStateManager.func_179098_w();
        }
    }

    @Override
    public void putInside(final View view)
    {
        super.putInside(view);

        //  Recompute scroll offset
        setSelectionEnd(selectionEnd);
    }

    @Override
    public void handleClick(final int mx, final int my)
    {
        if (mx < 0)
        {
            return;
        }

        final String visibleString = mc.field_71466_p.func_78269_a(text.substring(scrollOffset), getInternalWidth());
        final String trimmedString = mc.field_71466_p.func_78269_a(visibleString, mx);

        // Cache and restore scrollOffset when we change focus via click,
        // because onFocus() sets the cursor (and thus scroll offset) to the end.
        final int oldScrollOffset = scrollOffset;
        setFocus();
        scrollOffset = oldScrollOffset;
        setCursorPosition(trimmedString.length() + scrollOffset);
    }

    @Override
    public boolean onKeyTyped(final char c, final int key)
    {
        switch (c)
        {
            case 1:
                setCursorPosition(text.length());
                setSelectionEnd(0);
                return true;

            case 3:
                GuiScreen.func_146275_d(getSelectedText());
                return true;

            case 22:
                writeText(GuiScreen.func_146277_j());
                return true;

            case 24:
                GuiScreen.func_146275_d(getSelectedText());
                writeText("");
                return true;

            default:
                return handleKey(c, key);
        }
    }

    @Override
    public void onUpdate()
    {
        cursorBlinkCounter++;
    }

    /**
     * Write text into the field.
     *
     * @param str the string to write.
     */
    public void writeText(final String str)
    {
        final String filteredStr = filter.filter(str);

        final int insertAt = Math.min(cursorPosition, selectionEnd);
        final int insertEnd = Math.max(cursorPosition, selectionEnd);
        final int availableChars = (maxTextLength - text.length()) + (insertEnd - insertAt);

        @NotNull String result = "";
        if (text.length() > 0 && insertAt > 0)
        {
            result = text.substring(0, insertAt);
        }

        final int insertedLength;
        if (availableChars < filteredStr.length())
        {
            result = result + filteredStr.substring(0, availableChars);
            insertedLength = availableChars;
        }
        else
        {
            result = result + filteredStr;
            insertedLength = filteredStr.length();
        }

        if (text.length() > 0 && insertEnd < text.length())
        {
            result = result + text.substring(insertEnd);
        }

        text = result;
        moveCursorBy((insertAt - selectionEnd) + insertedLength);
    }

    /**
     * Delete an amount of words.
     *
     * @param count the amount.
     */
    public void deleteWords(final int count)
    {
        if (text.length() != 0)
        {
            if (selectionEnd != cursorPosition)
            {
                this.writeText("");
            }
            else
            {
                deleteFromCursor(this.getNthWordFromCursor(count) - this.cursorPosition);
            }
        }
    }

    /**
     * Delete amount of words from cursor.
     *
     * @param count the amount.
     */
    public void deleteFromCursor(final int count)
    {
        if (text.length() == 0)
        {
            return;
        }

        if (selectionEnd != cursorPosition)
        {
            this.writeText("");
        }
        else
        {
            final boolean backwards = count < 0;
            final int start = backwards ? (this.cursorPosition + count) : this.cursorPosition;
            final int end = backwards ? this.cursorPosition : (this.cursorPosition + count);
            @NotNull String result = "";

            if (start > 0)
            {
                result = text.substring(0, start);
            }

            if (end < text.length())
            {
                result = result + text.substring(end);
            }

            text = result;

            if (backwards)
            {
                this.moveCursorBy(count);
            }
        }
    }

    /**
     * Get the n'th word from a position.
     *
     * @param count the n.
     * @param pos   the position.
     * @return the length of the word.
     */
    public int getNthWordFromPos(final int count, final int pos)
    {
        final boolean reverse = count < 0;
        int position = pos;

        for (int i1 = 0; i1 < Math.abs(count); ++i1)
        {
            if (reverse)
            {
                while (position > 0 && text.charAt(position - 1) == ' ')
                {
                    --position;
                }
                while (position > 0 && text.charAt(position - 1) != ' ')
                {
                    --position;
                }
            }
            else
            {
                position = text.indexOf(' ', position);

                if (position == -1)
                {
                    position = text.length();
                }
                else
                {
                    while (position < text.length() && text.charAt(position) == ' ')
                    {
                        ++position;
                    }
                }
            }
        }

        return position;
    }

    /**
     * Get n'th word from cursor position.
     *
     * @param count the n.
     * @return the length.
     */
    public int getNthWordFromCursor(final int count)
    {
        return getNthWordFromPos(count, cursorPosition);
    }

    /**
     * Interface to filter words.
     */
    public interface Filter
    {
        /**
         * Apply the filter.
         *
         * @param s to the string.
         * @return the correct String.
         */
        String filter(String s);

        /**
         * Check if character is allowed.
         *
         * @param c character to test.
         * @return true if so.
         */
        boolean isAllowedCharacter(char c);
    }
}
