package com.minecolonies.coremod.colony;

import com.minecolonies.coremod.colony.buildings.AbstractBuilding;
import com.minecolonies.coremod.colony.buildings.AbstractBuildingWorker;
import com.minecolonies.coremod.colony.buildings.BuildingHome;
import com.minecolonies.coremod.colony.jobs.AbstractJob;
import com.minecolonies.coremod.configuration.Configurations;
import com.minecolonies.coremod.entity.EntityCitizen;
import com.minecolonies.coremod.entity.ai.basic.AbstractAISkeleton;
import com.minecolonies.coremod.util.BlockPosUtil;
import com.minecolonies.coremod.util.Log;
import io.netty.buffer.ByteBuf;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraftforge.fml.common.network.ByteBufUtils;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Random;

/**
 * Extra data for Citizens.
 */
public class CitizenData
{
    private static final float  MAX_HEALTH              = 20.0F;
    /**
     * Max level of an attribute a citizen may initially have.
     */
    private static final int    LEVEL_CAP               = 5;
    private static final int    LETTERS_IN_THE_ALPHABET = 26;
    /**
     * Tags.
     */
    private static final String TAG_ID                  = "id";
    private static final String TAG_NAME                = "name";
    private static final String TAG_FEMALE              = "female";
    private static final String TAG_TEXTURE             = "texture";
    private static final String TAG_LEVEL               = "level";
    private static final String TAG_EXPERIENCE          = "experience";
    private static final String TAG_HEALTH              = "health";
    private static final String TAG_MAX_HEALTH          = "maxHealth";
    private static final String TAG_SKILLS              = "skills";
    private static final String TAG_SKILL_STRENGTH      = "strength";
    private static final String TAG_SKILL_STAMINA       = "endurance";
    private static final String TAG_SKILL_SPEED         = "charisma";
    private static final String TAG_SKILL_INTELLIGENCE  = "intelligence";
    private static final String TAG_SKILL_DEXTERITY     = "dexterity";
    /**
     * The unique citizen id.
     */
    private final int                    id;
    private       String                 name;
    private       boolean                female;
    private       int                    textureId;
    private final Colony                 colony;
    @Nullable
    private       BuildingHome           homeBuilding;
    @Nullable
    private       AbstractBuildingWorker workBuilding;
    private       AbstractJob            job;
    private       boolean                dirty;
    //Citizen
    @Nullable
    private       EntityCitizen          entity;
    /**
     * Attributes, which influence the workers behaviour.
     * May be added more later.
     */
    private       int                    strength;
    private       int                    endurance;
    private       int                    charisma;
    private       int                    intelligence;
    private       int                    dexterity;
    private       double                 health;
    private       double                 maxHealth;
    /**
     * The current experience level the citizen is on.
     */
    private int level = 0;

    /**
     * The total amount of experience the citizen has.
     * This also includes the amount of experience within their Experience Bar.
     */
    private double experience;

    /**
     * Create a CitizenData given an ID.
     * Used as a super-constructor or during loading.
     *
     * @param id     ID of the Citizen.
     * @param colony Colony the Citizen belongs to.
     */
    public CitizenData(final int id, final Colony colony)
    {
        this.id = id;
        this.colony = colony;
    }

    /**
     * Creates CitizenData from tag compound.
     *
     * @param compound NBT compound to build from.
     * @param colony   Colony of the citizen.
     * @return CitizenData.
     */
    @NotNull
    public static CitizenData createFromNBT(@NotNull final NBTTagCompound compound, final Colony colony)
    {
        final int id = compound.func_74762_e(TAG_ID);
        final @NotNull CitizenData citizen = new CitizenData(id, colony);
        citizen.readFromNBT(compound);
        return citizen;
    }

    /**
     * Reads data from NBT-tag compound.
     *
     * @param compound NBT-Tag compound.
     */
    public void readFromNBT(@NotNull final NBTTagCompound compound)
    {
        name = compound.func_74779_i(TAG_NAME);
        female = compound.func_74767_n(TAG_FEMALE);
        textureId = compound.func_74762_e(TAG_TEXTURE);

        //  Attributes
        level = compound.func_74762_e(TAG_LEVEL);
        experience = compound.func_74762_e(TAG_EXPERIENCE);
        health = compound.func_74760_g(TAG_HEALTH);
        maxHealth = compound.func_74760_g(TAG_MAX_HEALTH);


        final NBTTagCompound nbtTagSkillsCompound = compound.func_74775_l("skills");
        strength = nbtTagSkillsCompound.func_74762_e("strength");
        endurance = nbtTagSkillsCompound.func_74762_e("endurance");
        charisma = nbtTagSkillsCompound.func_74762_e("charisma");
        intelligence = nbtTagSkillsCompound.func_74762_e("intelligence");
        dexterity = nbtTagSkillsCompound.func_74762_e("dexterity");

        if (compound.func_74764_b("job"))
        {
            setJob(AbstractJob.createFromNBT(this, compound.func_74775_l("job")));
        }
    }

    /**
     * Return the entity instance of the citizen data. Respawn the citizen if needed.
     *
     * @return {@link EntityCitizen} of the citizen data.
     */
    @Nullable
    public EntityCitizen getCitizenEntity()
    {
        return entity;
    }

    /**
     * Sets the entity of the citizen data.
     *
     * @param citizen {@link EntityCitizen} instance of the citizen data.
     */
    public void setCitizenEntity(final EntityCitizen citizen)
    {
        entity = citizen;
        markDirty();
    }

    /**
     * Marks the instance dirty.
     */
    public void markDirty()
    {
        dirty = true;
        colony.markCitizensDirty();
    }

    /**
     * Create a CitizenData View given it's saved NBTTagCompound.
     *
     * @param id  The citizen's id.
     * @param buf The network data.
     * @return View object of the citizen.
     */
    @Nullable
    public static CitizenDataView createCitizenDataView(final int id, final ByteBuf buf)
    {
        @Nullable CitizenDataView citizenDataView = new CitizenDataView(id);

        try
        {
            citizenDataView.deserialize(buf);
        }
        catch (final RuntimeException ex)
        {
            Log.getLogger().error(String.format("A CitizenData.View for #%d has thrown an exception during loading, its state cannot be restored. Report this to the mod author",
              citizenDataView.getID()), ex);
            citizenDataView = null;
        }

        return citizenDataView;
    }

    /**
     * Create a CitizenData given a CitizenEntity.
     *
     * @param entity Entity to initialize from.
     */
    public void initializeFromEntity(@NotNull final EntityCitizen entity)
    {
        final Random rand = entity.func_70681_au();

        this.entity = entity;

        //Assign the gender before name
        female = rand.nextBoolean();
        name = generateName(rand);

        textureId = entity.field_70170_p.field_73012_v.nextInt(Integer.MAX_VALUE);
        health = entity.func_110143_aJ();
        maxHealth = entity.func_110138_aP();
        experience = 0;
        level = 0;
        @NotNull final Random random = new Random();

        //Initialize the citizen skills and make sure they are never 0
        intelligence = random.nextInt(LEVEL_CAP - 1) + 1;
        charisma = random.nextInt(LEVEL_CAP - 1) + 1;
        strength = random.nextInt(LEVEL_CAP - 1) + 1;
        endurance = random.nextInt(LEVEL_CAP - 1) + 1;
        dexterity = random.nextInt(LEVEL_CAP - 1) + 1;

        markDirty();
    }

    /**
     * Generates a random name from a set of names.
     *
     * @param rand Random object.
     * @return Name of the citizen.
     */
    private String generateName(@NotNull final Random rand)
    {
        String citizenName;
        if (female)
        {
            citizenName = String.format("%s %s. %s", getRandomElement(rand, Configurations.femaleFirstNames), getRandomLetter(rand), getRandomElement(rand, Configurations.lastNames));
        }
        else
        {
            citizenName = String.format("%s %s. %s", getRandomElement(rand, Configurations.maleFirstNames), getRandomLetter(rand), getRandomElement(rand, Configurations.lastNames));
        }
        for (int i = 1; i <= this.getColony().getMaxCitizens(); i++)
        {
            if (this.getColony().getCitizen(i) != null && this.getColony().getCitizen(i).getName().equals(citizenName))
            {
                citizenName = generateName(rand);
            }
        }
        return citizenName;
    }

    /**
     * Returns a random element in a list.
     *
     * @param rand  Random object.
     * @param array Array to select from.
     * @return Random element from array.
     */
    private static String getRandomElement(@NotNull final Random rand, @NotNull final String[] array)
    {
        return array[rand.nextInt(array.length)];
    }

    /**
     * Returns a random capital letter from the alphabet.
     *
     * @param rand Random object.
     * @return Random capital letter.
     */
    private static char getRandomLetter(@NotNull final Random rand)
    {
        return (char) (rand.nextInt(LETTERS_IN_THE_ALPHABET) + 'A');
    }

    /**
     * Returns the id of the citizen.
     *
     * @return id of the citizen.
     */
    public int getId()
    {
        return id;
    }

    /**
     * Returns the colony of the citizen.
     *
     * @return colony of the citizen.
     */
    public Colony getColony()
    {
        return colony;
    }

    /**
     * Returns the name of the citizen.
     *
     * @return name of the citizen.
     */
    public String getName()
    {
        return name;
    }

    /**
     * Returns true if citizen is female, false for male.
     *
     * @return true for female, false for male.
     */
    public boolean isFemale()
    {
        return female;
    }

    /**
     * Returns the texture id for the citizen.
     *
     * @return texture ID.
     */
    public int getTextureId()
    {
        return textureId;
    }

    /**
     * Adds experience of the citizen.
     *
     * @param xp the amount of xp to add.
     */
    public void addExperience(final double xp)
    {
        this.experience += xp;
    }

    /**
     * Sets the level of the citizen.
     */
    public void increaseLevel()
    {
        this.level += 1;
    }

    /**
     * Returns whether or not the instance is dirty.
     *
     * @return true when dirty, otherwise false.
     */
    public boolean isDirty()
    {
        return dirty;
    }

    /**
     * Markt the instance not dirty.
     */
    public void clearDirty()
    {
        dirty = false;
    }

    /**
     * When a building is destroyed, inform the citizen so it can do any cleanup of associations that the building's.
     * own AbstractBuilding.onDestroyed did not do.
     *
     * @param building building that is destroyed.
     */
    public void onRemoveBuilding(final AbstractBuilding building)
    {
        if (getHomeBuilding() == building)
        {
            setHomeBuilding(null);
        }

        if (getWorkBuilding() == building)
        {
            setWorkBuilding(null);
        }
    }

    /**
     * Returns the home building of the citizen.
     *
     * @return home building.
     */
    @Nullable
    public BuildingHome getHomeBuilding()
    {
        return homeBuilding;
    }

    /**
     * Sets the home of the citizen.
     *
     * @param building home building.
     */
    public void setHomeBuilding(@Nullable final BuildingHome building)
    {
        if (homeBuilding != null && building != null && homeBuilding != building)
        {
            throw new IllegalStateException("CitizenData.setHomeBuilding() - already assigned a home building when setting a new home building");
        }
        else if (homeBuilding != building)
        {
            homeBuilding = building;
            markDirty();
        }
    }

    /**
     * Returns the work building of a citizen.
     *
     * @return home building of a citizen.
     */
    @Nullable
    public AbstractBuildingWorker getWorkBuilding()
    {
        return workBuilding;
    }

    /**
     * Sets the work building of a citizen.
     *
     * @param building work building.
     */
    public void setWorkBuilding(@Nullable final AbstractBuildingWorker building)
    {
        if (workBuilding != null && building != null && workBuilding != building)
        {
            throw new IllegalStateException("CitizenData.setWorkBuilding() - already assigned a work building when setting a new work building");
        }
        else if (workBuilding != building)
        {
            workBuilding = building;

            if (workBuilding != null)
            {
                //  We have a place to work, do we have the assigned Job?
                if (job == null)
                {
                    //  No job, create one!
                    setJob(workBuilding.createJob(this));
                    colony.getWorkManager().clearWorkForCitizen(this);
                }
            }
            else if (job != null)
            {
                final EntityCitizen citizen = getCitizenEntity();
                if (citizen != null)
                {
                    citizen.field_70714_bg.func_85156_a(citizen.field_70714_bg.field_75782_a.stream().filter(task -> task.field_75733_a instanceof AbstractAISkeleton).findFirst().orElse(null).field_75733_a);
                }
                //  No place of employment, get rid of our job
                setJob(null);
                colony.getWorkManager().clearWorkForCitizen(this);
            }

            markDirty();
        }
    }

    /**
     * Sets {@link EntityCitizen} to null for the instance.
     */
    public void clearCitizenEntity()
    {
        entity = null;
    }

    /**
     * Returns the job of the citizen.
     *
     * @return Job of the citizen.
     */
    public AbstractJob getJob()
    {
        return job;
    }

    /**
     * Sets the job of this citizen.
     *
     * @param job Job of the citizen.
     */
    public void setJob(final AbstractJob job)
    {
        this.job = job;

        @Nullable final EntityCitizen localEntity = getCitizenEntity();
        if (localEntity != null)
        {
            localEntity.onJobChanged(job);
        }

        markDirty();
    }

    /**
     * Returns the job subclass needed. Returns null on type mismatch.
     *
     * @param type the type of job wanted.
     * @param <J>  The job type returned.
     * @return the job this citizen has.
     */
    @Nullable
    public <J extends AbstractJob> J getJob(@NotNull final Class<J> type)
    {
        try
        {
            return type.cast(job);
        }
        catch (final ClassCastException ignored)
        {
            return null;
        }
    }

    /**
     * Writes the citiizen data to an NBT-compound.
     *
     * @param compound NBT-Tag compound.
     */
    public void writeToNBT(@NotNull final NBTTagCompound compound)
    {
        compound.func_74768_a(TAG_ID, id);
        compound.func_74778_a(TAG_NAME, name);
        compound.func_74757_a(TAG_FEMALE, female);
        compound.func_74768_a(TAG_TEXTURE, textureId);

        //  Attributes
        compound.func_74768_a(TAG_LEVEL, level);
        compound.func_74780_a(TAG_EXPERIENCE, experience);
        compound.func_74780_a(TAG_HEALTH, health);
        compound.func_74780_a(TAG_MAX_HEALTH, maxHealth);


        @NotNull final NBTTagCompound nbtTagSkillsCompound = new NBTTagCompound();
        nbtTagSkillsCompound.func_74768_a(TAG_SKILL_STRENGTH, strength);
        nbtTagSkillsCompound.func_74768_a(TAG_SKILL_STAMINA, endurance);
        nbtTagSkillsCompound.func_74768_a(TAG_SKILL_SPEED, charisma);
        nbtTagSkillsCompound.func_74768_a(TAG_SKILL_INTELLIGENCE, intelligence);
        nbtTagSkillsCompound.func_74768_a(TAG_SKILL_DEXTERITY, dexterity);
        compound.func_74782_a(TAG_SKILLS, nbtTagSkillsCompound);

        if (job != null)
        {
            @NotNull final NBTTagCompound jobCompound = new NBTTagCompound();
            job.writeToNBT(jobCompound);
            compound.func_74782_a("job", jobCompound);
        }
    }

    /**
     * Writes the citizen data to a byte buf for transition.
     *
     * @param buf Buffer to write to.
     */
    public void serializeViewNetworkData(@NotNull final ByteBuf buf)
    {
        ByteBufUtils.writeUTF8String(buf, name);
        buf.writeBoolean(female);

        buf.writeInt(entity != null ? entity.func_145782_y() : -1);

        buf.writeBoolean(homeBuilding != null);
        if (homeBuilding != null)
        {
            BlockPosUtil.writeToByteBuf(buf, homeBuilding.getID());
        }

        buf.writeBoolean(workBuilding != null);
        if (workBuilding != null)
        {
            BlockPosUtil.writeToByteBuf(buf, workBuilding.getID());
        }

        //  Attributes
        buf.writeInt(getLevel());
        buf.writeDouble(getExperience());

        //If entity is null assume the standard values as health
        if (entity == null)
        {
            buf.writeFloat(MAX_HEALTH);
            buf.writeFloat(MAX_HEALTH);
        }
        else
        {
            buf.writeFloat(entity.func_110143_aJ());
            buf.writeFloat(entity.func_110138_aP());
        }

        buf.writeInt(getStrength());
        buf.writeInt(getEndurance());
        buf.writeInt(getCharisma());
        buf.writeInt(getIntelligence());
        buf.writeInt(getDexterity());

        ByteBufUtils.writeUTF8String(buf, (job != null) ? job.getName() : "");
    }

    /**
     * Returns the level of the citizen.
     *
     * @return level of the citizen.
     */
    public int getLevel()
    {
        return level;
    }

    /**
     * Sets the level of the citizen.
     *
     * @param lvl the new level for the citizen.
     */
    public void setLevel(final int lvl)
    {
        this.level = lvl;
    }

    /**
     * Resets the experience and the experience level of the citizen.
     */
    public void resetExperienceAndLevel()
    {
        this.level = 0;
        this.experience = 0;
    }

    /**
     * Returns the experience of the citizen.
     *
     * @return experience of the citizen.
     */
    public double getExperience()
    {
        return experience;
    }

    /**
     * Strength getter.
     *
     * @return citizen Strength value.
     */
    public int getStrength()
    {
        return strength;
    }

    /**
     * Endurance getter.
     *
     * @return citizen Endurance value.
     */
    public int getEndurance()
    {
        return endurance;
    }

    /**
     * Charisma getter.
     *
     * @return citizen Charisma value.
     */
    public int getCharisma()
    {
        return charisma;
    }

    /**
     * Intelligence getter.
     *
     * @return citizen Intelligence value.
     */
    public int getIntelligence()
    {
        return intelligence;
    }

    /**
     * Dexterity getter.
     *
     * @return citizen Dexterity value.
     */
    public int getDexterity()
    {
        return dexterity;
    }
}
