package com.minecolonies.coremod.commands;

import com.minecolonies.coremod.colony.CitizenData;
import com.minecolonies.coremod.colony.Colony;
import com.minecolonies.coremod.colony.ColonyManager;
import com.minecolonies.coremod.colony.IColony;
import net.minecraft.command.CommandException;
import net.minecraft.command.ICommandSender;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.server.MinecraftServer;
import net.minecraft.util.math.BlockPos;
import net.minecraft.util.text.ITextComponent;
import net.minecraft.util.text.Style;
import net.minecraft.util.text.TextComponentString;
import net.minecraft.util.text.TextFormatting;
import net.minecraft.util.text.event.ClickEvent;
import org.jetbrains.annotations.NotNull;

import javax.annotation.Nullable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import static com.minecolonies.coremod.commands.AbstractSingleCommand.Commands.LISTCITIZENS;

/**
 * List all colonies.
 */
public class ListCitizensCommand extends AbstractSingleCommand
{

    public static final  String DESC                    = "list";
    private static final String CITIZEN_DESCRIPTION     = "§2ID: §f %d §2 Name: §f %s";
    private static final String COORDINATES_XYZ         = "§2Coordinates: §f §4x=§f%s §4y=§f%s §4z=§f%s";
    private static final String LIST_COMMAND_SUGGESTED  = "/mc citizens list %d %d";
    private static final String COMMAND_CITIZEN_INFO    = "/mc citizens info %s %s";
    private static final String PAGE_TOP                = "§2   ------------------ page %d of %d ------------------";
    private static final String PREV_PAGE               = " <- prev";
    private static final String NEXT_PAGE               = "next -> ";
    private static final String PAGE_LINE               = "§2 ----------------";
    private static final String PAGE_LINE_DIVIDER       = "§2 | ";
    private static final int    CITIZENS_ON_PAGE        = 9;

    /**
     * Initialize this SubCommand with it's parents.
     *
     * @param parents an array of all the parents.
     */
    public ListCitizensCommand(@NotNull final String... parents)
    {
        super(parents);
    }

    @NotNull
    @Override
    public String getCommandUsage(@NotNull final ICommandSender sender)
    {
        return super.getCommandUsage(sender) + "<colonyId>";
    }

    @Override
    public void execute(@NotNull final MinecraftServer server, @NotNull final ICommandSender sender, @NotNull final String... args) throws CommandException
    {
        int colonyId = getIthArgument(args, 0, getColonyId(sender));

        if(sender instanceof EntityPlayer)
        {
            if(colonyId == -1)
            {
                IColony colony = ColonyManager.getIColonyByOwner(sender.func_130014_f_(), (EntityPlayer) sender);
                if(colony != null)
                {
                    colonyId = colony.getID();
                }
            }

            final EntityPlayer player = (EntityPlayer) sender;
            if (!canPlayerUseCommand(player, LISTCITIZENS, colonyId))
            {
                sender.func_174793_f().func_145747_a(new TextComponentString("Not happenin bro!!, You are not permitted to do that!"));
                return;
            }
        }

        final Colony colony = ColonyManager.getColony(colonyId);

        final List<CitizenData> citizens = new ArrayList<>(colony.getCitizens().values());
        final int citizenCount = citizens.size();

        // check to see if we have to add one page to show the half page
        int page = getIthArgument(args, 1, 1);
        final int halfPage = (citizenCount % CITIZENS_ON_PAGE == 0) ? 0 : 1;
        final int pageCount = ((citizenCount) / CITIZENS_ON_PAGE) + halfPage;

        if (page < 1 || page > pageCount)
        {
            page = 1;
        }

        final int pageStartIndex = CITIZENS_ON_PAGE * (page - 1);
        final int pageStopIndex = Math.min(CITIZENS_ON_PAGE * page, citizenCount);

        final List<CitizenData> citizensPage;

        if (pageStartIndex < 0 || pageStartIndex >= citizenCount)
        {
            citizensPage = new ArrayList<>();
        }
        else
        {
            citizensPage = citizens.subList(pageStartIndex, pageStopIndex);
        }

        final ITextComponent headerLine = new TextComponentString(String.format(PAGE_TOP, page, pageCount));
        sender.func_145747_a(headerLine);

        for (final CitizenData citizen : citizensPage)
        {
            sender.func_145747_a(new TextComponentString(String.format(CITIZEN_DESCRIPTION,
              citizen.getId(),
              citizen.getName())).func_150255_a(new Style().func_150241_a(new ClickEvent(ClickEvent.Action.RUN_COMMAND,
                                                                                     String.format(COMMAND_CITIZEN_INFO, citizen.getColony().getID(), citizen.getId())))));

            if (citizen.getCitizenEntity() != null)
            {
                final BlockPos position = citizen.getCitizenEntity().func_180425_c();
                sender.func_145747_a(new TextComponentString(String.format(COORDINATES_XYZ, position.func_177958_n(), position.func_177956_o(), position.func_177952_p())));
            }
        }
        drawPageSwitcher(sender, page, citizenCount, halfPage, colonyId);
    }

    /**
     * Returns the colony of the owner or if not available colony 1.
     * First tries to get the IColony and then the Colony from the ColonyManager.
     *
     * @param sender the sender of the command.
     * @return the colonyId.
     */
    private static int getColonyId(@NotNull final ICommandSender sender)
    {
        final IColony tempColony = ColonyManager.getIColonyByOwner(sender.func_130014_f_(), sender.func_174793_f().func_110124_au());
        if (tempColony != null)
        {
            final Colony colony = ColonyManager.getColony(sender.func_130014_f_(), tempColony.getCenter());
            if (colony != null)
            {
                return colony.getID();
            }
        }

        return 1;
    }

    /**
     * Draws the page switcher at the bottom.
     *
     * @param sender   the sender.
     * @param page     the page number.
     * @param count    number of citizens.
     * @param halfPage the halfPage.
     * @param colonyId the colony id.
     */
    private static void drawPageSwitcher(@NotNull final ICommandSender sender, final int page, final int count, final int halfPage, final int colonyId)
    {
        final int prevPage = Math.max(0, page - 1);
        final int nextPage = Math.min(page + 1, (count / CITIZENS_ON_PAGE) + halfPage);

        final ITextComponent prevButton = new TextComponentString(PREV_PAGE).func_150255_a(new Style().func_150227_a(true).func_150238_a(TextFormatting.GOLD).func_150241_a(
          new ClickEvent(ClickEvent.Action.RUN_COMMAND, String.format(LIST_COMMAND_SUGGESTED, colonyId, prevPage))
        ));
        final ITextComponent nextButton = new TextComponentString(NEXT_PAGE).func_150255_a(new Style().func_150227_a(true).func_150238_a(TextFormatting.GOLD).func_150241_a(
          new ClickEvent(ClickEvent.Action.RUN_COMMAND, String.format(LIST_COMMAND_SUGGESTED, colonyId, nextPage))
        ));

        final ITextComponent beginLine = new TextComponentString(PAGE_LINE);
        final ITextComponent endLine = new TextComponentString(PAGE_LINE);
        sender.func_145747_a(beginLine.func_150257_a(prevButton).func_150257_a(new TextComponentString(PAGE_LINE_DIVIDER)).func_150257_a(nextButton).func_150257_a(endLine));
    }

    @NotNull
    @Override
    public List<String> getTabCompletionOptions(
                                                 @NotNull final MinecraftServer server,
                                                 @NotNull final ICommandSender sender,
                                                 @NotNull final String[] args,
                                                 @Nullable final BlockPos pos)
    {
        return Collections.emptyList();
    }

    @Override
    public boolean isUsernameIndex(@NotNull final String[] args, final int index)
    {
        return false;
    }
}
