package com.minecolonies.coremod.entity;

import com.minecolonies.coremod.util.MathUtils;
import net.minecraft.block.material.Material;
import net.minecraft.enchantment.EnchantmentHelper;
import net.minecraft.entity.Entity;
import net.minecraft.entity.item.EntityItem;
import net.minecraft.entity.item.EntityXPOrb;
import net.minecraft.init.SoundEvents;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.util.ResourceLocation;
import net.minecraft.util.math.AxisAlignedBB;
import net.minecraft.util.math.BlockPos;
import net.minecraft.util.math.MathHelper;
import net.minecraft.world.World;
import net.minecraft.world.WorldServer;
import net.minecraft.world.storage.loot.LootContext;
import net.minecraft.world.storage.loot.LootTableList;
import net.minecraftforge.fml.relauncher.Side;
import net.minecraftforge.fml.relauncher.SideOnly;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import static net.minecraft.util.EnumParticleTypes.*;

/**
 * Creates a custom fishHook for the Fisherman to throw.
 * This class manages the entity.
 */
public final class EntityFishHook extends Entity
{
    /**
     * Number of seconds to wait before removing a stuck hook.
     */
    private static final int TTL = 360;

    /**
     * Entity size to scale it down.
     */
    private static final float ENTITY_SIZE = 0.25F;

    /**
     * 180 degree used in trig. functions.
     */
    private static final double HALF_CIRCLE = 180.0;

    /**
     * Used as a speed multiplicator for drifting movement.
     */
    private static final double RANDOM_MOVEMENT_OFFSET = 0.007499999832361937;

    /**
     * Limits initial horizontal movement speed.
     */
    private static final double INITIAL_MOVEMENT_LIMITER = 0.16;

    /**
     * The hook starts a bit lower.
     */
    private static final double SUNKEN_OFFSET = 0.10000000149011612;

    /**
     * Multiplicator to get sum of edges.
     */
    private static final double NUM_BOUNDING_BOX_EDGES = 4.0;

    /**
     * factor to scale up to distance.
     */
    private static final double DISTANCE_FACTOR = 64.0;

    /**
     * Limits horizontal movement speed while bouncing.
     */
    private static final double BOUNCE_MOVEMENT_LIMITER = 0.2;

    /**
     * Limits horizontal movement speed while in air.
     */
    private static final float AIR_MOVEMENT_LIMITER = 0.92F;

    /**
     * Limits horizontal movement speed while on the ground.
     */
    private static final double GROUND_MOVEMENT_LIMITER = 0.5;

    /**
     * Limits horizontal movement speed while in the water.
     */
    private static final double WATER_MOVEMENT_LIMITER = 0.03999999910593033;

    /**
     * Chance to slow down fishing while the sky is not visible.
     */
    private static final double NO_CLEAR_SKY_CHANCE = 0.5;

    /**
     * Chance to get rare drops while fishing. Higher value leads to a lower chance.
     */
    private static final int INCREASE_RARENESS_MODIFIER = 200;

    /**
     * The citizen who threw this rod.
     */
    private EntityCitizen citizen;

    /**
     * The fishing speed enchantment level on the rod that threw this hook.
     */
    private int fishingSpeedEnchantment;

    /**
     * The fishing loot enchantment level on the rod that threw this hook.
     */
    private int fishingLootEnchantment;

    /**
     * If this hook is in the ground.
     */
    private boolean inGround;

    /**
     * A counter for at what position in the shaking movement the hook is.
     */
    private int    shake;
    private int    countdownNoFish;
    private int    countdownFishNear;
    private int    countdownFishBites;
    private double relativeRotation;

    /**
     * entity creation time.
     * Used to check it the hook got stuck.
     */
    private final long creationTime;

    /**
     * When a fish is on the hook, this will be true.
     */
    private boolean isFishCaugth = false;

    /**
     * Constructor for throwing out a hook.
     *
     * @param world   the world the hook lives in.
     * @param citizen the citizen throwing the hook.
     */
    public EntityFishHook(final World world, @NotNull final EntityCitizen citizen)
    {
        this(world);
        this.citizen = citizen;
        this.func_70012_b(citizen.field_70165_t,
          citizen.field_70163_u + 1.62 - citizen.func_70033_W(),
          citizen.field_70161_v,
          citizen.field_70177_z,
          citizen.field_70125_A);
        this.field_70165_t -= Math.cos(this.field_70177_z / HALF_CIRCLE * Math.PI) * INITIAL_MOVEMENT_LIMITER;
        this.field_70163_u -= SUNKEN_OFFSET;
        this.field_70161_v -= Math.sin(this.field_70177_z / HALF_CIRCLE * Math.PI) * INITIAL_MOVEMENT_LIMITER;
        this.func_70107_b(this.field_70165_t, this.field_70163_u, this.field_70161_v);
        final double f = 0.4;
        this.field_70159_w = -Math.sin(this.field_70177_z / HALF_CIRCLE * Math.PI) * Math.cos(this.field_70125_A / HALF_CIRCLE * Math.PI) * f;
        this.field_70179_y = Math.cos(this.field_70177_z / HALF_CIRCLE * Math.PI) * Math.cos(this.field_70125_A / HALF_CIRCLE * Math.PI) * f;
        this.field_70181_x = -Math.sin(this.field_70125_A / HALF_CIRCLE * Math.PI) * f;
        this.setPosition(this.field_70159_w, this.field_70181_x, this.field_70179_y, 1.5, 1.0);
        fishingSpeedEnchantment = EnchantmentHelper.func_151387_h(citizen);
        fishingLootEnchantment = EnchantmentHelper.func_151386_g(citizen);
    }

    /**
     * Lowest denominator constructor.
     * Used by other constructors to do general stuff.
     *
     * @param world the world this entity lives in.
     */
    public EntityFishHook(final World world)
    {
        super(world);
        this.func_70105_a(ENTITY_SIZE, ENTITY_SIZE);
        this.field_70158_ak = true;
        this.creationTime = System.nanoTime();
        fishingLootEnchantment = 0;
        fishingSpeedEnchantment = 0;
    }

    private void setPosition(final double x, final double y, final double z, final double yaw, final double pitch)
    {
        final double squareRootXYZ = MathHelper.func_76133_a(x * x + y * y + z * z);
        double newX = x / squareRootXYZ;
        double newY = y / squareRootXYZ;
        double newZ = z / squareRootXYZ;
        newX += this.field_70146_Z.nextGaussian() * RANDOM_MOVEMENT_OFFSET * pitch;
        newY += this.field_70146_Z.nextGaussian() * RANDOM_MOVEMENT_OFFSET * pitch;
        newZ += this.field_70146_Z.nextGaussian() * RANDOM_MOVEMENT_OFFSET * pitch;
        newX *= yaw;
        newY *= yaw;
        newZ *= yaw;
        this.field_70159_w = newX;
        this.field_70181_x = newY;
        this.field_70179_y = newZ;
        this.field_70126_B = this.field_70177_z = (float) (Math.atan2(newX, newZ) * HALF_CIRCLE / Math.PI);
        this.field_70127_C = this.field_70125_A = (float) (Math.atan2(newY, Math.sqrt(newX * newX + newZ * newZ)) * HALF_CIRCLE / Math.PI);
    }

    /**
     * Returns the citizen throwing the hook.
     *
     * @return a citizen.
     */
    public EntityCitizen getCitizen()
    {
        return citizen;
    }

    /**
     * Minecraft may call this method.
     */
    @Override
    protected void func_70088_a()
    {
        /**
         * No need to use this method.
         * It will be ignored.
         */
    }

    @Override
    public boolean equals(@Nullable final Object o)
    {
        return !(o == null || getClass() != o.getClass() || !super.equals(o));
    }

    @Override
    public int hashCode()
    {
        return func_145782_y();
    }

    /**
     * Called to update the entity's position/logic.
     */
    @Override
    public void func_70071_h_()
    {
        super.func_70071_h_();
        if (fishHookIsOverTimeToLive())
        {
            this.func_70106_y();
        }
        if (bounceFromGround() || this.inGround)
        {
            return;
        }

        moveSomeStuff();
    }

    /**
     * Checks if the entity is in range to render by using the past in distance and comparing it to its average edge.
     * length * 64 * renderDistanceWeight Args: distance.
     *
     * @param range the real range.
     * @return true or false.
     */
    @Override
    @SideOnly(Side.CLIENT)
    public boolean func_70112_a(final double range)
    {
        double maxLength = this.func_174813_aQ().func_72320_b() * NUM_BOUNDING_BOX_EDGES;
        maxLength *= DISTANCE_FACTOR;
        return range < maxLength * maxLength;
    }

    /**
     * If a hook gets loaded, kill it immediately.
     */
    @Override
    public void func_70037_a(final NBTTagCompound unused)
    {
        this.func_70106_y();
    }

    /**
     * No need to write anything to NBT.
     * A hook does not need to be saved.
     */
    @Override
    public void func_70014_b(final NBTTagCompound unused)
    {
        //We don't save this
    }

    /**
     * Sets the velocity to the args.
     *
     * @param vectorX directionX
     * @param vectorY directionY
     * @param vectorZ directionZ
     */
    @Override
    @SideOnly(Side.CLIENT)
    public void func_70016_h(final double vectorX, final double vectorY, final double vectorZ)
    {
        this.field_70159_w = vectorX;
        this.field_70181_x = vectorY;
        this.field_70179_y = vectorZ;
    }

    /**
     * Check if a fishhook is there for too long.
     * and became bugged.
     * After 360 seconds remove the hook.
     *
     * @return true if the time to live is over
     */
    public boolean fishHookIsOverTimeToLive()
    {
        return MathUtils.nanoSecondsToSeconds(System.nanoTime() - creationTime) > TTL;
    }

    /**
     * Update some movement things for the hook.
     * Detect if the hook is on ground and maybe bounce.
     * Also count how long the hook is laying on the ground or in water.
     *
     * @return true if the hook is killed.
     */
    private boolean bounceFromGround()
    {
        if (this.shake > 0)
        {
            --this.shake;
        }

        if (!this.inGround)
        {
            return false;
        }

        this.inGround = false;
        this.field_70159_w *= (this.field_70146_Z.nextDouble() * BOUNCE_MOVEMENT_LIMITER);
        this.field_70181_x *= (this.field_70146_Z.nextDouble() * BOUNCE_MOVEMENT_LIMITER);
        this.field_70179_y *= (this.field_70146_Z.nextDouble() * BOUNCE_MOVEMENT_LIMITER);

        return false;
    }

    /**
     * Main update method thingy.
     * hopefully I'm able to reduce it somewhat...
     */
    private void moveSomeStuff()
    {
        updateMotionAndRotation();
        double movementLimiter = AIR_MOVEMENT_LIMITER;

        if (this.field_70122_E || this.field_70123_F)
        {
            movementLimiter = GROUND_MOVEMENT_LIMITER;
        }

        final byte numSteps = 5;
        double waterDensity = 0.0;

        //Check how much water is around the hook
        for (int j = 0; j < numSteps; ++j)
        {
            final double d3 = this.func_174813_aQ().field_72338_b + (this.func_174813_aQ().field_72337_e - this.func_174813_aQ().field_72338_b) * j / numSteps;
            final double d4 = this.func_174813_aQ().field_72338_b + (this.func_174813_aQ().field_72337_e - this.func_174813_aQ().field_72338_b) * (j + 1) / numSteps;

            @NotNull final AxisAlignedBB axisAlignedBB1 = new AxisAlignedBB(
                                                                             this.func_174813_aQ().field_72340_a,
                                                                             d3,
                                                                             this.func_174813_aQ().field_72339_c,
                                                                             this.func_174813_aQ().field_72336_d,
                                                                             d4,
                                                                             this.func_174813_aQ().field_72334_f);

            //If the hook is swimming
            if (this.field_70170_p.func_72830_b(axisAlignedBB1, Material.field_151586_h))
            {
                waterDensity += 1.0 / numSteps;
            }
        }

        checkIfFishBites(waterDensity);

        final double currentDistance = waterDensity * 2.0D - 1.0;
        this.field_70181_x += WATER_MOVEMENT_LIMITER * currentDistance;

        if (waterDensity > 0.0)
        {
            movementLimiter *= 0.9;
            this.field_70181_x *= 0.8;
        }

        this.field_70159_w *= movementLimiter;
        this.field_70181_x *= movementLimiter;
        this.field_70179_y *= movementLimiter;
        this.func_70107_b(this.field_70165_t, this.field_70163_u, this.field_70161_v);
    }

    /**
     * Update the fishing hooks motion.
     * and its rotation.
     */
    private void updateMotionAndRotation()
    {
        this.func_70091_d(this.field_70159_w, this.field_70181_x, this.field_70179_y);
        final double motion = Math.sqrt(this.field_70159_w * this.field_70159_w + this.field_70179_y * this.field_70179_y);
        this.field_70177_z = (float) (Math.atan2(this.field_70181_x, this.field_70179_y) * HALF_CIRCLE / Math.PI);
        this.field_70125_A = (float) (Math.atan2(this.field_70181_x, motion) * HALF_CIRCLE / Math.PI);
        while ((double) this.field_70125_A - (double) this.field_70127_C < -HALF_CIRCLE)
        {
            this.field_70127_C -= 360.0;
        }

        while ((double) this.field_70125_A - (double) this.field_70127_C >= HALF_CIRCLE)
        {
            this.field_70127_C += 360.0;
        }

        while ((double) this.field_70177_z - (double) this.field_70126_B < -HALF_CIRCLE)
        {
            this.field_70126_B -= 360.0;
        }

        while ((double) this.field_70177_z - (double) this.field_70126_B >= HALF_CIRCLE)
        {
            this.field_70126_B += 360.0;
        }

        this.field_70125_A = (float) ((double) this.field_70127_C + ((double) this.field_70125_A - (double) this.field_70127_C) * 0.2D);
        this.field_70177_z = (float) ((double) this.field_70126_B + ((double) this.field_70177_z - (double) this.field_70126_B) * 0.2D);
    }

    /**
     * Server side method to do.
     * some animation and movement stuff.
     * when the hook swims in water.
     * <p>
     * will set isFishCaught if a fish bites.
     *
     * @param waterDensity the amount of water around.
     */
    private void checkIfFishBites(final double waterDensity)
    {
        if (!this.field_70170_p.field_72995_K && waterDensity > 0.0)
        {
            int fishingProgressStep = 1;

            if (this.field_70146_Z.nextDouble() < NO_CLEAR_SKY_CHANCE
                  && !this.field_70170_p.func_175710_j(new BlockPos(MathHelper.func_76128_c(this.field_70165_t), MathHelper.func_76128_c(this.field_70163_u) + 1, MathHelper.func_76128_c(this.field_70161_v))))
            {
                --fishingProgressStep;
            }

            if (this.countdownNoFish > 0)
            {
                updateNoFishCounter();
                return;
            }

            @NotNull final WorldServer worldServer = (WorldServer) this.field_70170_p;

            if (this.countdownFishNear > 0)
            {
                renderBubble(fishingProgressStep, worldServer);
                return;
            }

            if (this.countdownFishBites > 0)
            {
                this.countdownFishBites -= fishingProgressStep;

                renderFishBiteOrSwim(worldServer);
            }
            else
            {
                this.countdownFishNear = MathHelper.func_76136_a(this.field_70146_Z, 100, 900);
                this.countdownFishNear -= fishingSpeedEnchantment * 20 * 5;
            }
        }
    }

    private void updateNoFishCounter()
    {
        --this.countdownNoFish;

        if (this.countdownNoFish <= 0)
        {
            this.countdownFishNear = 0;
            this.countdownFishBites = 0;
        }
        else
        {
            this.field_70181_x -= (this.field_70146_Z.nextDouble() * this.field_70146_Z.nextDouble() * this.field_70146_Z.nextDouble()) * BOUNCE_MOVEMENT_LIMITER;
        }
    }

    private void renderBubble(final int fishingProgressStep, @NotNull final WorldServer worldServer)
    {
        this.countdownFishNear -= fishingProgressStep;

        double bubbleY = 0.15;

        if (this.countdownFishNear < 20)
        {
            bubbleY = bubbleY + (double) (20 - this.countdownFishNear) * 0.05;
        }
        else if (this.countdownFishNear < 40)
        {
            bubbleY = bubbleY + (double) (40 - this.countdownFishNear) * 0.02;
        }
        else if (this.countdownFishNear < 60)
        {
            bubbleY = bubbleY + (double) (60 - this.countdownFishNear) * 0.01;
        }

        if (this.field_70146_Z.nextDouble() < bubbleY)
        {
            renderLittleSplash(worldServer);
        }

        if (this.countdownFishNear <= 0)
        {
            this.relativeRotation = MathHelper.func_151240_a(this.field_70146_Z, 0.0F, 360.0F);
            this.countdownFishBites = MathHelper.func_76136_a(this.field_70146_Z, 20, 80);
        }
    }

    private void renderFishBiteOrSwim(@NotNull final WorldServer worldServer)
    {
        if (this.countdownFishBites <= 0)
        {
            //Show fish bite animation
            showFishBiteAnimation(worldServer);
        }
        else
        {
            //Show fish swim animation
            showFishSwimmingTowardsHookAnimation(worldServer);
        }
    }

    /**
     * Render little splashes around the fishing hook.
     * simulating fish movement.
     *
     * @param worldServer the server side world.
     */
    private void renderLittleSplash(@NotNull final WorldServer worldServer)
    {
        final double sinYPosition = (double) MathHelper.func_151240_a(this.field_70146_Z, 0.0F, 360.0F) * 0.017453292D;
        final double cosYPosition = MathHelper.func_151240_a(this.field_70146_Z, 25.0F, 60.0F);
        final double bubbleX = this.field_70165_t + (Math.sin(sinYPosition) * cosYPosition * 0.1);
        final double increasedYPosition = Math.floor(this.func_174813_aQ().field_72338_b) + 1.0;
        final double bubbleZ = this.field_70161_v + (Math.cos(sinYPosition) * cosYPosition * 0.1);
        worldServer.func_175739_a(WATER_SPLASH,
          bubbleX,
          increasedYPosition,
          bubbleZ,
          2 + this.field_70146_Z.nextInt(2),
          SUNKEN_OFFSET,
          0.0,
          SUNKEN_OFFSET,
          0.0);
    }

    /**
     * Show bubbles towards the hook.
     * Let the hook sink in a bit.
     * Play a sound to signal to the player,
     * that a fish bit.
     *
     * @param worldServer the server side world.
     */
    private void showFishBiteAnimation(@NotNull final WorldServer worldServer)
    {
        this.field_70181_x -= 0.20000000298023224D;
        this.func_184185_a(SoundEvents.field_187609_F, ENTITY_SIZE, (float) (1.0D + this.field_70146_Z.nextGaussian() * 0.4D));
        final double bubbleY = Math.floor(this.func_174813_aQ().field_72338_b);
        worldServer.func_175739_a(WATER_BUBBLE,
          this.field_70165_t,
          bubbleY + 1.0,
          this.field_70161_v,
          (int) (1.0 + this.field_70130_N * 20.0),
          (double) this.field_70130_N,
          0.0,
          (double) this.field_70130_N,
          0.20000000298023224);
        worldServer.func_175739_a(WATER_WAKE,
          this.field_70165_t,
          bubbleY + 1.0,
          this.field_70161_v,
          (int) (1.0 + this.field_70130_N * 20.0),
          (double) this.field_70130_N,
          0.0,
          (double) this.field_70130_N,
          0.20000000298023224);
        this.countdownNoFish = MathHelper.func_76136_a(this.field_70146_Z, 10, 30);
        isFishCaugth = true;
    }

    /**
     * Show bubbles moving towards the hook.
     * make it look like a fish will bite soon.
     *
     * @param worldServer the server side world.
     */
    private void showFishSwimmingTowardsHookAnimation(@NotNull final WorldServer worldServer)
    {
        this.relativeRotation = this.relativeRotation + this.field_70146_Z.nextGaussian() * NUM_BOUNDING_BOX_EDGES;
        final double bubbleY = this.relativeRotation * 0.017453292;
        final double sinYPosition = Math.sin(bubbleY);
        final double cosYPosition = Math.cos(bubbleY);
        final double bubbleX = this.field_70165_t + (sinYPosition * this.countdownFishBites * 0.1);
        final double increasedYPosition = Math.floor(this.func_174813_aQ().field_72338_b) + 1.0;
        final double bubbleZ = this.field_70161_v + (cosYPosition * this.countdownFishBites * 0.1);

        if (this.field_70146_Z.nextDouble() < 0.15)
        {
            worldServer.func_175739_a(WATER_BUBBLE, bubbleX, increasedYPosition - SUNKEN_OFFSET, bubbleZ, 1, sinYPosition, 0.1D, cosYPosition, 0.0);
        }

        final double f3 = sinYPosition * 0.04;
        final double f4 = cosYPosition * 0.04;
        worldServer.func_175739_a(WATER_WAKE, bubbleX, increasedYPosition, bubbleZ, 0, f4, 0.01, -f3, 1.0);
        worldServer.func_175739_a(WATER_WAKE, bubbleX, increasedYPosition, bubbleZ, 0, -f4, 0.01, f3, 1.0);
    }

    /**
     * Returns a damage value by how much the fishingRod should be damaged.
     * Also spawns loot and exp and destroys the hook.
     *
     * @param citizen the fisherman fishing.
     * @return the number of damage points to be deducted.
     */
    public int getDamage(@NotNull final EntityCitizen citizen)
    {
        if (this.field_70170_p.field_72995_K)
        {
            this.func_70106_y();
            return 0;
        }
        byte itemDamage = 0;
        if (isFishCaugth)
        {
            if (this.countdownNoFish > 0)
            {
                spawnLootAndExp(citizen);
                itemDamage = 1;
            }

            if (this.inGround)
            {
                itemDamage = 0;
            }
        }
        this.func_70106_y();
        return itemDamage;
    }

    /**
     * Spawns a random loot from the loot table.
     * and some exp orbs.
     * Should be called when retrieving a hook.
     * todo: Perhaps streamline this and directly add the items?
     *
     * @param citizen the fisherman getting the loot.
     */
    private void spawnLootAndExp(@NotNull final EntityCitizen citizen)
    {
        final double citizenPosX = citizen.field_70165_t;
        final double citizenPosY = citizen.field_70163_u;
        final double citizenPosZ = citizen.field_70161_v;
        @NotNull final EntityItem entityitem = new EntityItem(this.field_70170_p, this.field_70165_t, this.field_70163_u, this.field_70161_v, this.getFishingLoot(citizen));
        final double distanceX = citizenPosX - this.field_70165_t;
        final double distanceY = citizenPosY - this.field_70163_u;
        final double distanceZ = citizenPosZ - this.field_70161_v;

        entityitem.field_70159_w = distanceX * 0.1;
        entityitem.field_70181_x = distanceY * 0.1 + Math.sqrt(Math.sqrt(distanceX * distanceX + distanceY * distanceY + distanceZ * distanceZ)) * 0.08;
        entityitem.field_70179_y = distanceZ * 0.1;
        this.field_70170_p.func_72838_d(entityitem);
        citizen.field_70170_p.func_72838_d(new EntityXPOrb(citizen.field_70170_p,
                                                             citizenPosX,
                                                             citizenPosY + 0.D,
                                                             citizenPosZ + 0.5,
                                                             this.field_70146_Z.nextInt(6) + 1));
    }

    /**
     * Determines which loot table should be used.
     * <p>
     * The selection is somewhat random and depends on enchantments.
     * and the level of the fisherman hut.
     *
     * @param citizen the fisherman getting the loot.
     * @return an ItemStack randomly from the loot table.
     */
    private ItemStack getFishingLoot(final EntityCitizen citizen)
    {
        //Reduce random to get more fish drops
        final int random = this.field_70170_p.field_73012_v.nextInt(INCREASE_RARENESS_MODIFIER);
        final int buildingLevel = citizen.getWorkBuilding().getBuildingLevel();
        //Cut to minimum value of 0.
        final int lootBonus = MathHelper.func_76125_a(fishingLootEnchantment- fishingSpeedEnchantment,0,Integer.MAX_VALUE);

        if (random >= buildingLevel * (lootBonus + 1) || buildingLevel == 1)
        {
            if (random >= INCREASE_RARENESS_MODIFIER - buildingLevel * (lootBonus + 1) && buildingLevel >= 2)
            {
                return getLootForLootTable(LootTableList.field_186388_am);
            }

            return getLootForLootTable(LootTableList.field_186390_ao);
        }
        else
        {
            return getLootForLootTable(LootTableList.field_186389_an);
        }
    }

    /**
     * Return some random loot of a defined lootTable.
     * @param lootTable the lootTable.
     * @return the ItemStack of the loot.
     */
    private ItemStack getLootForLootTable(ResourceLocation lootTable)
    {
        final LootContext.Builder lootContextBuilder = new LootContext.Builder((WorldServer) this.field_70170_p);
        for (final ItemStack itemstack : this.field_70170_p.func_184146_ak()
                .func_186521_a(lootTable)
                .func_186462_a(this.field_70146_Z, lootContextBuilder.func_186471_a()))
        {
            return itemstack;
        }
        return null;
    }

    /**
     * returns true if a fish was caught.
     *
     * @return true | false
     */
    public boolean caughtFish()
    {
        return isFishCaugth;
    }
}
