package com.minecolonies.coremod.entity.ai.citizen.guard;

import com.minecolonies.coremod.colony.jobs.JobGuard;
import com.minecolonies.coremod.entity.ai.util.AIState;
import com.minecolonies.coremod.entity.ai.util.AITarget;
import com.minecolonies.coremod.util.InventoryFunctions;
import com.minecolonies.coremod.util.Utils;
import net.minecraft.enchantment.EnchantmentHelper;
import net.minecraft.entity.EntityLivingBase;
import net.minecraft.init.SoundEvents;
import net.minecraft.item.ItemStack;
import net.minecraft.item.ItemSword;
import net.minecraft.util.DamageSource;
import net.minecraft.util.EnumHand;
import org.jetbrains.annotations.NotNull;

import static com.minecolonies.coremod.entity.ai.util.AIState.*;

/**
 * Handles the AI of the guard entities.
 */
public class EntityAIMeleeGuard extends AbstractEntityAIGuard
{
    /**
     * Basic delay for the next shot.
     */
    private static final int BASE_RELOAD_TIME = 30;

    /**
     * The pitch will be divided by this to calculate it for the arrow sound.
     */
    private static final double PITCH_DIVIDER = 1.0D;

    /**
     * The base pitch, add more to this to change the sound.
     */
    private static final double BASE_PITCH = 0.8D;

    /**
     * Random is multiplied by this to get a random arrow sound.
     */
    private static final double PITCH_MULTIPLIER = 0.4D;

    /**
     * Quantity to be moved to rotate the entity without actually moving.
     */
    private static final double MOVE_MINIMAL = 0.01D;

    /**
     * Quantity the worker should turn around all at once.
     */
    private static final double TURN_AROUND = 180D;

    /**
     * Normal volume at which sounds are played at.
     */
    private static final double BASIC_VOLUME = 1.0D;

    /**
     * Experience the guard receives each shot arrow.
     */
    private static final double XP_EACH_HIT = 0.2;

    /**
     * Base speed of the guard he follows his target.
     */
    private static final int BASE_FOLLOW_SPEED = 1;

    /**
     * Base multiplier increasing the attack speed each level.
     */
    private static final double BASE_FOLLOW_SPEED_MULTIPLIER = 0.25D;

    /**
     * The Min distance of the guard to attack entities.
     */
    private static final double MIN_ATTACK_DISTANCE = 2.0D;

    /**
     * Damage per range attack.
     */
    private static final double DAMAGE_PER_ATTACK = 0.5;

    /**
     * Chance that a mob is lit on fire when a weapon has the fire aspect enchantment.
     */
    private static final int FIRE_CHANCE_MULTIPLIER = 4;

    /**
     * Sets up some important skeleton stuff for every ai.
     *
     * @param job the job class
     */
    public EntityAIMeleeGuard(@NotNull final JobGuard job)
    {
        super(job);
        super.registerTargets(
          new AITarget(GUARD_SEARCH_TARGET, this::searchTarget),
          new AITarget(GUARD_GET_TARGET, this::getTarget),
          new AITarget(GUARD_HUNT_DOWN_TARGET, this::huntDown),
          new AITarget(GUARD_PATROL, this::patrol),
          new AITarget(GUARD_RESTOCK, this::goToBuilding)
        );

        if (worker.getCitizenData() != null)
        {
            worker.setSkillModifier(2 * worker.getCitizenData().getStrength() + worker.getCitizenData().getEndurance());
            worker.func_98053_h(true);
        }
    }

    @Override
    protected AIState searchTarget()
    {
        if (checkForWeapon())
        {
            return AIState.GUARD_SEARCH_TARGET;
        }
        InventoryFunctions.matchFirstInProvider(worker, stack -> stack != null && Utils.doesItemServeAsWeapon(stack), worker::setHeldItem);
        return super.searchTarget();
    }

    /**
     * Follow the target and kill it.
     *
     * @return the next AIState.
     */
    protected AIState huntDown()
    {
        if (!targetEntity.func_70089_S() || checkForWeapon())
        {
            targetEntity = null;
            worker.func_70659_e((float) 1.0D);
            return AIState.GUARD_GATHERING;
        }

        if (worker.func_70635_at().func_75522_a(targetEntity) && worker.func_70032_d(targetEntity) <= MIN_ATTACK_DISTANCE)
        {
            worker.func_184602_cy();
            attackEntity(targetEntity, (float) DAMAGE_PER_ATTACK);
            setDelay(getReloadTime());
            attacksExecuted += 1;
            currentSearchDistance = START_SEARCH_DISTANCE;

            if (attacksExecuted >= getMaxAttacksUntilRestock())
            {
                return AIState.GUARD_RESTOCK;
            }

            return AIState.GUARD_HUNT_DOWN_TARGET;
        }

        if (shouldReturnToTarget(targetEntity.func_180425_c(), FOLLOW_RANGE))
        {
            return AIState.GUARD_PATROL;
        }

        worker.func_70659_e((float) (BASE_FOLLOW_SPEED + BASE_FOLLOW_SPEED_MULTIPLIER * worker.getExperienceLevel()));
        worker.isWorkerAtSiteWithMove(targetEntity.func_180425_c(), (int) MIN_ATTACK_DISTANCE);

        return AIState.GUARD_SEARCH_TARGET;
    }

    private void attackEntity(@NotNull final EntityLivingBase entityToAttack, final float baseDamage)
    {
        double damgeToBeDealt = baseDamage;

        if (worker.func_110143_aJ() <= 2)
        {
            damgeToBeDealt *= 2;
        }

        final ItemStack heldItem = worker.func_184586_b(EnumHand.MAIN_HAND);
        if (heldItem != null)
        {
            if (heldItem.func_77973_b() instanceof ItemSword)
            {
                damgeToBeDealt += ((ItemSword) heldItem.func_77973_b()).func_150931_i();
            }
            damgeToBeDealt += EnchantmentHelper.func_152377_a(heldItem, targetEntity.func_70668_bt());
        }

        targetEntity.func_70097_a(new DamageSource(worker.func_70005_c_()), (float) damgeToBeDealt);
        targetEntity.func_70604_c(worker);

        final int fireAspectModifier = EnchantmentHelper.func_90036_a(worker);
        if (fireAspectModifier > 0)
        {
            targetEntity.func_70015_d(fireAspectModifier * FIRE_CHANCE_MULTIPLIER);
        }

        worker.addExperience(XP_EACH_HIT);
        if (targetEntity.func_110143_aJ() <= 0.0F)
        {
            this.onKilledEntity(targetEntity);
        }

        worker.func_70625_a(entityToAttack, (float) TURN_AROUND, (float) TURN_AROUND);
        worker.func_70671_ap().func_75651_a(entityToAttack, (float) TURN_AROUND, (float) TURN_AROUND);

        final double xDiff = targetEntity.field_70165_t - worker.field_70165_t;
        final double zDiff = targetEntity.field_70161_v - worker.field_70161_v;

        final double goToX = xDiff > 0 ? MOVE_MINIMAL : -MOVE_MINIMAL;
        final double goToZ = zDiff > 0 ? MOVE_MINIMAL : -MOVE_MINIMAL;

        worker.func_70091_d(goToX, 0, goToZ);

        worker.func_184609_a(EnumHand.MAIN_HAND);
        worker.func_184185_a(SoundEvents.field_187730_dW, (float) BASIC_VOLUME, (float) getRandomPitch());

        worker.damageItemInHand(1);
    }

    private int getReloadTime()
    {
        return BASE_RELOAD_TIME / (worker.getExperienceLevel() + 1);
    }

    private double getRandomPitch()
    {
        return PITCH_DIVIDER / (worker.func_70681_au().nextDouble() * PITCH_MULTIPLIER + BASE_PITCH);
    }
}
