package com.minecolonies.coremod.entity.ai.minimal;

import net.minecraft.block.Block;
import net.minecraft.block.BlockFenceGate;
import net.minecraft.block.material.Material;
import net.minecraft.block.state.IBlockState;
import net.minecraft.entity.EntityLiving;
import net.minecraft.entity.ai.EntityAIBase;
import net.minecraft.pathfinding.Path;
import net.minecraft.pathfinding.PathNavigateGround;
import net.minecraft.pathfinding.PathPoint;
import net.minecraft.util.math.BlockPos;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

/**
 * Used for gate interaction for the citizens.
 */
public class EntityAIGateInteract extends EntityAIBase
{
    /**
     * Number of blocks to check for the fence gate - height.
     */
    private static final int    HEIGHT_TO_CHECK = 2;
    /**
     * Number of blocks to check for the fence gate - length.
     */
    private static final int    LENGTH_TO_CHECK = 2;
    /**
     * The length of half a block.
     */
    private static final double HALF_BLOCK      = 0.5D;
    /**
     * The min distance the gate has to be from the citizen.
     */
    private static final double MIN_DISTANCE    = 2.25D;
    /**
     * Our citizen.
     */
    protected EntityLiving   theEntity;
    /**
     * The gate position.
     */
    protected BlockPos       gatePosition;
    /**
     * The gate block.
     */
    @Nullable
    protected BlockFenceGate gateBlock;
    /**
     * Check if the interaction with the fenceGate stopped already.
     */
    private   boolean        hasStoppedFenceInteraction;
    /**
     * The entities x position.
     */
    private   double         entityPositionX;
    /**
     * The entities z position.
     */
    private   double         entityPositionZ;

    /**
     * Constructor called to register the AI class with an entity.
     *
     * @param entityIn the registering entity.
     */
    public EntityAIGateInteract(@NotNull final EntityLiving entityIn)
    {
        super();
        this.gatePosition = BlockPos.field_177992_a;
        this.theEntity = entityIn;
        if (!(entityIn.func_70661_as() instanceof PathNavigateGround))
        {
            throw new IllegalArgumentException("Unsupported mob type for DoorInteractGoal");
        }
    }

    /**
     * Checks if the Interaction should be executed.
     *
     * @return true or false depending on the conditions.
     */
    @Override
    public boolean func_75250_a()
    {
        return this.theEntity.field_70123_F && checkPath();
    }

    /**
     * Checks if there exists a path.
     *
     * @return true if the fence gate can be passed.
     */
    private boolean checkPath()
    {
        @NotNull final PathNavigateGround pathnavigateground = (PathNavigateGround) this.theEntity.func_70661_as();
        final Path path = pathnavigateground.func_75505_d();
        return path != null && !path.func_75879_b() && pathnavigateground.func_179686_g() && checkFenceGate(path);
    }

    /**
     * Checks if the citizen is close enough to an existing fence gate.
     *
     * @param path the path through the fence.
     * @return true if the gate can be passed
     */
    private boolean checkFenceGate(@NotNull final Path path)
    {
        final int maxLengthToCheck = Math.min(path.func_75873_e() + LENGTH_TO_CHECK, path.func_75874_d());
        for (int i = 0; i < maxLengthToCheck; ++i)
        {
            final PathPoint pathpoint = path.func_75877_a(i);
            for (int level = 0; level < HEIGHT_TO_CHECK; level++)
            {
                this.gatePosition = new BlockPos(pathpoint.field_75839_a, pathpoint.field_75837_b + level, pathpoint.field_75838_c);
                if (this.theEntity.func_70092_e((double) this.gatePosition.func_177958_n(), this.theEntity.field_70163_u, (double) this.gatePosition.func_177952_p()) <= MIN_DISTANCE)
                {
                    this.gateBlock = this.getBlockFence(this.gatePosition);
                    if (this.gateBlock != null)
                    {
                        return true;
                    }
                }
            }
        }

        this.gatePosition = (new BlockPos(this.theEntity)).func_177984_a();
        this.gateBlock = this.getBlockFence(this.gatePosition);
        return this.gateBlock != null;
    }

    /**
     * Returns a fenceBlock if available.
     *
     * @param pos the position to be searched.
     * @return fenceBlock or null.
     */
    private BlockFenceGate getBlockFence(@NotNull final BlockPos pos)
    {
        final IBlockState blockState = this.theEntity.field_70170_p.func_180495_p(pos);
        Block block = blockState.func_177230_c();
        if (!(block instanceof BlockFenceGate && blockState.func_185904_a() == Material.field_151575_d))
        {
            block = this.theEntity.field_70170_p.func_180495_p(this.theEntity.func_180425_c()).func_177230_c();
            gatePosition = this.theEntity.func_180425_c();
        }
        return block instanceof BlockFenceGate && blockState.func_185904_a() == Material.field_151575_d ? (BlockFenceGate) block : null;
    }

    /**
     * Checks if the execution is still ongoing.
     *
     * @return true or false.
     */
    @Override
    public boolean func_75253_b()
    {
        return !this.hasStoppedFenceInteraction;
    }

    /**
     * Starts the execution.
     */
    @Override
    public void func_75249_e()
    {
        this.hasStoppedFenceInteraction = false;
        this.entityPositionX = this.gatePosition.func_177958_n() + HALF_BLOCK - this.theEntity.field_70165_t;
        this.entityPositionZ = this.gatePosition.func_177952_p() + HALF_BLOCK - this.theEntity.field_70161_v;
    }

    /**
     * Updates the task and checks if the citizen passed the gate already.
     */
    @Override
    public void func_75246_d()
    {
        final double entityDistX = this.gatePosition.func_177958_n() + HALF_BLOCK - this.theEntity.field_70165_t;
        final double entityDistZ = this.gatePosition.func_177952_p() + HALF_BLOCK - this.theEntity.field_70161_v;
        final double totalDist = this.entityPositionX * entityDistX + this.entityPositionZ * entityDistZ;
        if (totalDist < 0.0D)
        {
            this.hasStoppedFenceInteraction = true;
        }
    }
}
