package com.minecolonies.structures.helpers;

import com.minecolonies.coremod.blocks.ModBlocks;
import com.minecolonies.coremod.lib.Constants;
import com.minecolonies.coremod.util.BlockUtils;
import com.minecolonies.coremod.util.Log;
import com.minecolonies.structures.fake.FakeEntity;
import com.minecolonies.structures.fake.FakeWorld;
import com.minecolonies.structures.lib.ModelHolder;
import net.minecraft.block.Block;
import net.minecraft.block.state.IBlockState;
import net.minecraft.client.Minecraft;
import net.minecraft.client.renderer.GlStateManager;
import net.minecraft.client.renderer.RenderHelper;
import net.minecraft.client.renderer.Tessellator;
import net.minecraft.client.renderer.VertexBuffer;
import net.minecraft.client.renderer.block.model.BakedQuad;
import net.minecraft.client.renderer.block.model.IBakedModel;
import net.minecraft.client.renderer.texture.TextureMap;
import net.minecraft.client.renderer.tileentity.TileEntityRendererDispatcher;
import net.minecraft.entity.Entity;
import net.minecraft.entity.EntityList;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.nbt.CompressedStreamTools;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.server.MinecraftServer;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.*;
import net.minecraft.util.math.BlockPos;
import net.minecraft.util.math.Vec3d;
import net.minecraft.world.World;
import net.minecraft.world.gen.structure.template.PlacementSettings;
import net.minecraft.world.gen.structure.template.Template;
import net.minecraftforge.client.ForgeHooksClient;
import net.minecraftforge.client.MinecraftForgeClient;
import net.minecraftforge.client.model.pipeline.LightUtil;
import net.minecraftforge.fml.common.FMLCommonHandler;
import org.apache.commons.io.IOUtils;
import org.jetbrains.annotations.NotNull;
import org.lwjgl.opengl.GL11;

import javax.annotation.Nullable;
import java.io.*;
import java.util.List;

/**
 * Structure class, used to store, create, get structures.
 */
public class Structure
{
    /**
     * Rotation by 90°.
     */
    private static final double NINETY_DEGREES = 90D;

    /**
     * Rotation by 270°.
     */
    private static final double TWO_HUNDRED_SEVENTY_DEGREES = 270D;

    /**
     * Rotation by 180°.
     */
    private static final double ONE_HUNDED_EIGHTY_DEGREES = 270D;

    /**
     * Used for scale.
     */
    private static final double SCALE                    = 1.001;

    /**
     * Template of the structure.
     */
    private Template          template;
    private Minecraft         mc;
    private PlacementSettings settings;

    /**
     * Constuctor of Structure, tries to create a new structure.
     *
     * @param world         with world.
     * @param structureName name of the structure (at stored location).
     * @param settings      it's settings.
     */
    public Structure(@Nullable final World world, final String structureName, final PlacementSettings settings)
    {
        InputStream inputstream = MinecraftServer.class.getResourceAsStream("/assets/" + Constants.MOD_ID + "/schematics/" + structureName + ".nbt");

        if (world == null || world.field_72995_K)
        {
            this.settings = settings;
            this.mc = Minecraft.func_71410_x();
        }

        //Might be at a different location!
        if (inputstream == null)
        {
            try
            {
                final File decorationFolder;

                if (FMLCommonHandler.instance().getMinecraftServerInstance() == null)
                {
                    decorationFolder = new File(Minecraft.func_71410_x().field_71412_D, "minecolonies/");
                }
                else
                {
                    decorationFolder = new File(FMLCommonHandler.instance().getMinecraftServerInstance().func_71238_n(), "minecolonies/");
                }
                if(decorationFolder.exists())
                {
                    inputstream = new FileInputStream(decorationFolder.getPath() + "/" + structureName + ".nbt");
                }
                else
                {
                    throw new FileNotFoundException("Unable to find structure: " + structureName);
                }
            }
            catch (final FileNotFoundException e)
            {
                Log.getLogger().warn("Couldn't find any structure with this name anywhere", e);
            }
        }

        if (inputstream == null)
        {
            return;
        }

        try
        {
            this.template = readTemplateFromStream(inputstream);
        }
        catch (final IOException e)
        {
            Log.getLogger().warn(String.format("Failed to load template %s", structureName), e);
        }
        finally
        {
            IOUtils.closeQuietly(inputstream);
        }
    }

    /**
     * Reads a template from an inputstream.
     */
    private static Template readTemplateFromStream(final InputStream stream) throws IOException
    {
        final NBTTagCompound nbttagcompound = CompressedStreamTools.func_74796_a(stream);
        final Template template = new Template();
        template.func_186256_b(nbttagcompound);
        return template;
    }

    /**
     * Checks if the template is null.
     *
     * @return true if the template is null.
     */
    public boolean isTemplateMissing()
    {
        return template == null;
    }

    public Template.BlockInfo[] getBlockInfo()
    {
        Template.BlockInfo[] blockList = new Template.BlockInfo[template.field_186270_a.size()];
        blockList = template.field_186270_a.toArray(blockList);
        return blockList;
    }

    /**
     * Get entity array at position in world.
     *
     * @param world the world.
     * @param pos   the position.
     * @return the entity array.
     */
    public Entity[] getEntityInfo(final World world, final BlockPos pos)
    {
        Template.EntityInfo[] entityInfoList = new Template.EntityInfo[template.field_186271_b.size()];
        entityInfoList = template.field_186270_a.toArray(entityInfoList);

        final Entity[] entityList = null;

        for (int i = 0; i < entityInfoList.length; i++)
        {
            final Entity finalEntity = EntityList.func_75615_a(entityInfoList[i].field_186249_c, world);
            final Vec3d entityVec = entityInfoList[i].field_186247_a.func_178787_e(new Vec3d(pos));
            finalEntity.func_70107_b(entityVec.field_72450_a, entityVec.field_72448_b, entityVec.field_72449_c);
        }

        return entityList;
    }

    /**
     * Get size of structure.
     *
     * @param rotation with rotation.
     * @return size as blockPos (x = length, z = width, y = height).
     */
    public BlockPos getSize(final Rotation rotation)
    {
        return this.template.func_186257_a(rotation);
    }

    public void setPlacementSettings(final PlacementSettings settings)
    {
        this.settings = settings;
    }

    /**
     * Renders the structure.
     *
     * @param startingPos  the start pos to render.
     * @param clientWorld  the world of the client.
     * @param player       the player object.
     * @param partialTicks the partial ticks.
     */
    public void renderStructure(@NotNull final BlockPos startingPos, @NotNull final World clientWorld, @NotNull final EntityPlayer player, final float partialTicks)
    {
        final Template.BlockInfo[] blockList = this.getBlockInfoWithSettings(this.settings);
        final Entity[] entityList = this.getEntityInfoWithSettings(clientWorld, startingPos, this.settings);

        for (final Template.BlockInfo aBlockList : blockList)
        {
            Block block = aBlockList.field_186243_b.func_177230_c();
            IBlockState iblockstate = aBlockList.field_186243_b;

            if(block == ModBlocks.blockSubstitution)
            {
                continue;
            }

            if(block == ModBlocks.blockSolidSubstitution)
            {
                iblockstate = BlockUtils.getSubstitutionBlockAtWorld(clientWorld, startingPos);
                block = iblockstate.func_177230_c();
            }

            final BlockPos blockpos = aBlockList.field_186242_a.func_177971_a(startingPos);
            final IBlockState iBlockExtendedState = block.getExtendedState(iblockstate, clientWorld, blockpos);
            final IBakedModel ibakedmodel = Minecraft.func_71410_x().func_175602_ab().func_184389_a(iblockstate);
            TileEntity tileentity = null;
            if (block.hasTileEntity(iblockstate) && aBlockList.field_186244_c != null)
            {
                tileentity = block.createTileEntity(clientWorld, iblockstate);
                tileentity.func_145839_a(aBlockList.field_186244_c);
            }
            final ModelHolder models = new ModelHolder(blockpos, iblockstate, iBlockExtendedState, tileentity, ibakedmodel);
            getQuads(models, models.quads);
            this.renderGhost(clientWorld, models, player, partialTicks);
        }

        for (final Entity anEntityList : entityList)
        {
            if(anEntityList != null)
            {
                Minecraft.func_71410_x().func_175598_ae().func_188388_a(anEntityList, 0.0F, true);
            }
        }
    }

    /**
     * Get blockInfo of structure with a specific setting.
     *
     * @param settings the setting.
     * @return the block info array.
     */
    public Template.BlockInfo[] getBlockInfoWithSettings(final PlacementSettings settings)
    {
        Template.BlockInfo[] blockList = new Template.BlockInfo[template.field_186270_a.size()];
        blockList = template.field_186270_a.toArray(blockList);

        for (int i = 0; i < blockList.length; i++)
        {
            final IBlockState finalState = blockList[i].field_186243_b.func_185902_a(settings.func_186212_b()).func_185907_a(settings.func_186215_c());
            final BlockPos finalPos = Template.func_186266_a(settings, blockList[i].field_186242_a);
            final Template.BlockInfo finalInfo = new Template.BlockInfo(finalPos, finalState, blockList[i].field_186244_c);
            blockList[i] = finalInfo;
        }
        return blockList;
    }

    /**
     * Get entity info with specific setting.
     *
     * @param entityInfo the entity to transform.
     * @param world      world the entity is in.
     * @param pos        the position it is at.
     * @param settings   the settings.
     * @return the entity info aray.
     */
    public Template.EntityInfo transformEntityInfoWithSettings(final Template.EntityInfo entityInfo, final World world, final BlockPos pos, final PlacementSettings settings)
    {
        final Entity finalEntity = EntityList.func_75615_a(entityInfo.field_186249_c, world);

        //err might be here? only use pos? or don't add?
        final Vec3d entityVec = Structure.transformedVec3d(settings, entityInfo.field_186247_a).func_178787_e(new Vec3d(pos));

        if (finalEntity != null)
        {
            finalEntity.field_70126_B = (float) (finalEntity.func_184217_a(settings.func_186212_b()) - NINETY_DEGREES);
            final double rotationYaw
                    = (double)finalEntity.func_184217_a(settings.func_186212_b()) + ((double)finalEntity.field_70177_z - (double)finalEntity.func_184229_a(settings.func_186215_c()));

            finalEntity.func_70012_b(entityVec.field_72450_a, entityVec.field_72448_b, entityVec.field_72449_c,
                    (float) rotationYaw, finalEntity.field_70125_A);

            final NBTTagCompound nbttagcompound = new NBTTagCompound();
            finalEntity.func_70039_c(nbttagcompound);
            return new Template.EntityInfo(entityInfo.field_186247_a, entityInfo.field_186248_b, nbttagcompound);
        }

        return null;
    }


    /**
     * Transforms the entity's current yaw with the given Rotation and returns it. This does not have a side-effect.
     * @param transformRotation the incoming rotation.
     * @param previousYaw the previous rotation yaw.
     * @return the new rotation yaw.
     */
    public double getRotatedYaw(Rotation transformRotation, double previousYaw)
    {
        switch (transformRotation)
        {
            case CLOCKWISE_180:
                return previousYaw + NINETY_DEGREES;
            case COUNTERCLOCKWISE_90:
                return previousYaw + TWO_HUNDRED_SEVENTY_DEGREES;
            case CLOCKWISE_90:
                return previousYaw + ONE_HUNDED_EIGHTY_DEGREES;
            default:
                return previousYaw;
        }
    }

    /**
     * Get entity info with specific setting.
     *
     * @param world    world the entity is in.
     * @param pos      the position it is at.
     * @param settings the settings.
     * @return the entity info aray.
     */
    public Entity[] getEntityInfoWithSettings(final World world, final BlockPos pos, final PlacementSettings settings)
    {
        Template.EntityInfo[] entityInfoList = new Template.EntityInfo[template.field_186271_b.size()];
        entityInfoList = template.field_186271_b.toArray(entityInfoList);

        final Entity[] entityList = new Entity[entityInfoList.length];

        for (int i = 0; i < entityInfoList.length; i++)
        {
            final Entity finalEntity = EntityList.func_75615_a(entityInfoList[i].field_186249_c, world);
            final Vec3d entityVec = Structure.transformedVec3d(settings, entityInfoList[i].field_186247_a).func_178787_e(new Vec3d(pos));

            if (finalEntity != null)
            {
                finalEntity.field_70126_B = (float) (finalEntity.func_184217_a(settings.func_186212_b()) - NINETY_DEGREES);
                final double rotation =
                        (double) finalEntity.func_184217_a(settings.func_186212_b()) + ((double) finalEntity.field_70177_z - finalEntity.func_184229_a(settings.func_186215_c()));
                finalEntity.func_70012_b(entityVec.field_72450_a, entityVec.field_72448_b, entityVec.field_72449_c, (float) rotation, finalEntity.field_70125_A);
            }
            entityList[i] = finalEntity;
        }

        return entityList;
    }

    private static void getQuads(final ModelHolder holder, final List<BakedQuad> quads)
    {
        if (holder.actualState.func_185901_i() == EnumBlockRenderType.MODEL)
        {
            final BlockRenderLayer originalLayer = MinecraftForgeClient.getRenderLayer();

            for (final BlockRenderLayer layer : BlockRenderLayer.values())
            {
                if (holder.actualState.func_177230_c().canRenderInLayer(holder.actualState, layer))
                {
                    ForgeHooksClient.setRenderLayer(layer);

                    for (final EnumFacing facing : EnumFacing.values())
                    {
                        quads.addAll(holder.model.func_188616_a(holder.extendedState, facing, 0));
                    }

                    quads.addAll(holder.model.func_188616_a(holder.extendedState, null, 0));
                }
            }

            ForgeHooksClient.setRenderLayer(originalLayer);
        }
    }

    private void renderGhost(final World world, final ModelHolder holder, final EntityPlayer player, final float partialTicks)
    {
        final boolean existingModel = !this.mc.field_71441_e.func_175623_d(holder.pos);

        final IBlockState actualState = holder.actualState;
        final Block block = actualState.func_177230_c();

        if (actualState.func_185901_i() == EnumBlockRenderType.MODEL)
        {
            final BlockRenderLayer originalLayer = MinecraftForgeClient.getRenderLayer();

            for (final BlockRenderLayer layer : BlockRenderLayer.values())
            {
                if (block.canRenderInLayer(actualState, layer))
                {
                    this.mc.func_110434_K().func_110577_a(TextureMap.field_110575_b);
                    ForgeHooksClient.setRenderLayer(layer);
                    this.renderGhostBlock(world, holder, player, layer, existingModel, partialTicks);
                    holder.setRendered(true);
                }
            }

            ForgeHooksClient.setRenderLayer(originalLayer);
        }

        if (holder.te != null && !holder.isRendered())
        {
            final TileEntity te = holder.te;
            te.func_174878_a(holder.pos);
            final FakeWorld fakeWorld = new FakeWorld(holder.actualState, world.func_72860_G(), world.func_72912_H(), world.field_73011_w, world.field_72984_F, true);
            te.func_145834_a(fakeWorld);
            final int pass = 0;

            if (te.shouldRenderInPass(pass))
            {
                final TileEntityRendererDispatcher terd = TileEntityRendererDispatcher.field_147556_a;
                terd.func_190056_a(fakeWorld,
                        Minecraft.func_71410_x().field_71446_o,
                        Minecraft.func_71410_x().field_71466_p,
                        new FakeEntity(fakeWorld),
                        null,
                        0.0F);
                GL11.glPushMatrix();
                terd.field_147553_e = Minecraft.func_71410_x().field_71446_o;
                terd.preDrawBatch();
                GL11.glColor4f(1F, 1F, 1F, 1F);
                terd.func_180546_a(te, partialTicks, -1);
                terd.drawBatch(pass);
                GL11.glPopMatrix();
            }
        }
    }

    /**
     * Transform a Vec3d with placement settings.
     *
     * @param settings the settings.
     * @param vec      the vector.
     * @return the new vector.
     */
    public static Vec3d transformedVec3d(final PlacementSettings settings, final Vec3d vec)
    {
        final Mirror mirrorIn = settings.func_186212_b();
        final Rotation rotationIn = settings.func_186215_c();
        double xCoord = vec.field_72450_a;
        final double yCoord = vec.field_72448_b;
        double zCoord = vec.field_72449_c;
        boolean flag = true;

        switch (mirrorIn)
        {
            case LEFT_RIGHT:
                zCoord = 1.0D - zCoord;
                break;
            case FRONT_BACK:
                xCoord = 1.0D - xCoord;
                break;
            default:
                flag = false;
        }

        switch (rotationIn)
        {
            case COUNTERCLOCKWISE_90:
                return new Vec3d(zCoord, yCoord, 1.0D - xCoord);
            case CLOCKWISE_90:
                return new Vec3d(1.0D - zCoord, yCoord, xCoord);
            case CLOCKWISE_180:
                return new Vec3d(1.0D - xCoord, yCoord, 1.0D - zCoord);
            default:
                return flag ? new Vec3d(xCoord, yCoord, zCoord) : vec;
        }
    }

    private void renderGhostBlock(
            final World world,
            final ModelHolder holder,
            final EntityPlayer player,
            final BlockRenderLayer layer,
            final boolean existingModel,
            final float partialTicks)
    {
        final double dx = player.field_70142_S + (player.field_70165_t - player.field_70142_S) * partialTicks;
        final double dy = player.field_70137_T + (player.field_70163_u - player.field_70137_T) * partialTicks;
        final double dz = player.field_70136_U + (player.field_70161_v - player.field_70136_U) * partialTicks;
        final BlockPos pos = holder.pos;

        GlStateManager.func_179094_E();
        GlStateManager.func_179137_b(pos.func_177958_n() - dx, pos.func_177956_o() - dy, pos.func_177952_p() - dz);

        if (existingModel)
        {
            GlStateManager.func_179139_a(SCALE, SCALE, SCALE);
        }

        RenderHelper.func_74518_a();

        if (layer == BlockRenderLayer.CUTOUT)
        {
            this.mc.func_110434_K().func_110581_b(TextureMap.field_110575_b).func_174936_b(false, false);
        }

        GlStateManager.func_179131_c(1F, 1F, 1F, 1F);

        final int alpha = ((int) (1.0D * 0xFF)) << 24;

        GlStateManager.func_179147_l();
        GlStateManager.func_179098_w();

        GlStateManager.func_179112_b(GL11.GL_SRC_ALPHA, GL11.GL_ONE_MINUS_SRC_ALPHA);
        GlStateManager.func_179135_a(false, false, false, false);
        this.renderModel(world, holder, pos, alpha);

        GlStateManager.func_179135_a(true, true, true, true);
        GlStateManager.func_179143_c(GL11.GL_LEQUAL);
        this.renderModel(world, holder, pos, alpha);

        GlStateManager.func_179084_k();

        if (layer == BlockRenderLayer.CUTOUT)
        {
            this.mc.func_110434_K().func_110581_b(TextureMap.field_110575_b).func_174935_a();
        }

        GlStateManager.func_179121_F();
    }

    private void renderModel(final World world, final ModelHolder holder, final BlockPos pos, final int alpha)
    {
        for (final EnumFacing facing : EnumFacing.values())
        {
            this.renderQuads(world, holder.actualState, pos, holder.model.func_188616_a(holder.extendedState, facing, 0), alpha);
        }

        this.renderQuads(world, holder.actualState, pos, holder.model.func_188616_a(holder.extendedState, null, 0), alpha);
    }

    private void renderQuads(final World world, final IBlockState actualState, final BlockPos pos, final List<BakedQuad> quads, final int alpha)
    {
        final Tessellator tessellator = Tessellator.func_178181_a();
        final VertexBuffer buffer = tessellator.func_178180_c();

        for (final BakedQuad quad : quads)
        {
            buffer.func_181668_a(GL11.GL_QUADS, quad.getFormat());

            final int color = quad.func_178212_b() ? this.getTint(world, actualState, pos, alpha, quad.func_178211_c()) : (alpha | 0xffffff);

            LightUtil.renderQuadColor(buffer, quad, color);

            tessellator.func_78381_a();
        }
    }

    private int getTint(final World world, final IBlockState actualState, final BlockPos pos, final int alpha, final int tintIndex)
    {
        return alpha | this.mc.func_184125_al().func_186724_a(actualState, world, pos, tintIndex);
    }

    /**
     * Get all additional entities.
     *
     * @return list of entities.
     */
    public List<Template.EntityInfo> getTileEntities()
    {
        return template.field_186271_b;
    }

    /**
     * Get the Placement settings of the structure.
     *
     * @return the settings.
     */
    public PlacementSettings getSettings()
    {
        return settings;
    }
}
